#---------------------------------------------------------------------------------------
# Bundle: KLayoutAna3.app
#
# Type: Automator wrapper app (Launcher)
#
# Purpose: Launch klayout.app within the Anaconda3 environment with
#          "Anaconda3-2025.06-0-MacOSX-[arm64|x86_64].pkg".
#
# Descriptions:
#  This script invokes "klayout.app" that shares the Anaconda3 development environment.
#  It uses Qt6, Ruby3.4, and Python3.13 from a per-arch virtual environment:
#    [arm64]  /opt/anaconda3/envs/klayout-qt6
#    [x86_64] /Applications/anaconda3/envs/klayout-qt6
#---------------------------------------------------------------------------------------
umask 022

#--------------------------------------------------------------------------
# Bash safe-guard
#--------------------------------------------------------------------------
set -euo pipefail
IFS=$'\n\t'

#--------------------------------------------------------------------------
# Fail logging
#--------------------------------------------------------------------------
LOG_DIR="${HOME}/Library/Logs/AppLaunchers"
mkdir -p "${LOG_DIR}"
LOG_FILE="${LOG_DIR}/KLayoutAna3.launch.log"
log() {
  local IFS=' '
  printf '%s %s\n' "$(date '+%F %T')" "$*" | tee -a "${LOG_FILE}"
}
trap '
  ec=$?
  if (( ec == 0 )); then
    log "EXIT 0 (success)"
  else
    log "EXIT ${ec} at line ${LINENO}"
  fi
  exit "${ec}"
' EXIT

#--------------------------------------------------------------------------
# Identify the machine architecture
#--------------------------------------------------------------------------
ARCH="$(uname -m)"
case "$ARCH" in
  arm64)  ARCH_TAG="arm64-native"   ;;
  x86_64) ARCH_TAG="x86_64-native"  ;;
  *) echo "Unsupported architecture: ${ARCH}" >&2; exit 1 ;;
esac

#--------------------------------------------------------------------------
# Localizing the language
# Workaround discussed in https://github.com/KLayout/klayout/issues/1213
#
# If you want to unify the locale (e.g. English UI),
# export LANG="en_US.UTF-8"; export LC_ALL="$LANG"
#--------------------------------------------------------------------------
APPLE_LOCALE="$(defaults read -g AppleLocale 2>/dev/null || true)"
if [[ -n "${APPLE_LOCALE}" ]]; then
  export LANG="${APPLE_LOCALE}.UTF-8"
else
  export LANG="en_US.UTF-8"
fi

#--------------------------------------------------------------------------
# Your setups if required
#--------------------------------------------------------------------------

#--------------------------------------------------------------------------
# Export the 'KLAYOUT_GIT_HTTP_PROXY' environment variable if you need to
# use the git-based Salt Package Manager through a proxy server.
#--------------------------------------------------------------------------
#export KLAYOUT_GIT_HTTP_PROXY="http://111.222.333.444:5678"

#--------------------------------------------------------------------------
# Downloading data from the package server might time out (default=10 sec).
# If so, set the 'KLAYOUT_HTTP_TIMEOUT' environment variable.
# For example, to make a timeout of 20 seconds,
#--------------------------------------------------------------------------
#export KLAYOUT_HTTP_TIMEOUT=20.0

#--------------------------------------------------------------------------
# Anaconda virtual environment per-arch
# Prepend Anaconda bin so helper tools (python, pip) resolve consistently.
#--------------------------------------------------------------------------
if [[ "${ARCH_TAG}" == "arm64-native" ]]; then
  VIRTUAL_ENV="/opt/anaconda3/envs/klayout-qt6"
  export PATH="/opt/anaconda3/bin:$PATH"
else
  VIRTUAL_ENV="/Applications/anaconda3/envs/klayout-qt6"
  export PATH="/Applications/anaconda3/bin:$PATH"
fi

if [[ ! -d "${VIRTUAL_ENV}" ]]; then
  echo "Virtual env not found: ${VIRTUAL_ENV}" >&2
  exit 2
fi
export VIRTUAL_ENV

#--------------------------------------------------------------------------
# Export the 'KLAYOUT_PYTHONPATH' environment variable.
# Ref. https://www.klayout.de/forum/discussion/2557/
#
# Do NOT set KLAYOUT_PYTHONHOME to avoid interpreter conflicts.
#--------------------------------------------------------------------------
PYVER_DIR="$(printf '%s\n' "${VIRTUAL_ENV}/lib/python3."* 2>/dev/null | sort -V | tail -n1 || true)"
if [[ -n "${PYVER_DIR}" && -d "${PYVER_DIR}/site-packages" ]]; then
  export KLAYOUT_PYTHONPATH="${PYVER_DIR}/site-packages"
else
  # fallback
  if [[ -d "${VIRTUAL_ENV}/lib/python3.13/site-packages" ]]; then
    export KLAYOUT_PYTHONPATH="${VIRTUAL_ENV}/lib/python3.13/site-packages"
  else
    echo "Cannot determine site-packages under ${VIRTUAL_ENV}/lib" >&2
    exit 3
  fi
fi

#--------------------------------------------------------------------------
# App bundle / config path
#--------------------------------------------------------------------------
KLAYOUT_APP="/Applications/klayout.app"
KLAYOUT_RC="$HOME/.klayout/klayoutrc"

if [[ ! -d "${KLAYOUT_APP}" ]]; then
  echo "KLayout app bundle not found: ${KLAYOUT_APP}" >&2
  exit 4
fi

#--------------------------------------------------------------------------
# Set launch options
#   - With "-n" option, you can invoke multiple instances.
#--------------------------------------------------------------------------
MULTI_INSTANCE=true
# true: launch a new instance each time
# false: focus or reuse an existing instance

# Set OPEN_FLAGS
OPEN_FLAGS=(-a "${KLAYOUT_APP}")
if [[ "${MULTI_INSTANCE}" == true ]]; then
  OPEN_FLAGS=(-n "${OPEN_FLAGS[@]}")
fi

# Set KLayout options
KLAYOUT_ARGS=(-e) # editor mode
if [[ -f "${KLAYOUT_RC}" ]]; then
  KLAYOUT_ARGS+=(-c "${KLAYOUT_RC}")
fi

# Set Qt options
QT_ARGS=(-style fusion)

log "---------------------------------------------------------------------"
log "Launch attempt on $(date '+%F %T')"
log "BEGIN user=$(id -un) arch=${ARCH_TAG} pid=$$"
log "ARCH=${ARCH_TAG}  VENV=${VIRTUAL_ENV}"
log "KLAYOUT_PYTHONPATH=${KLAYOUT_PYTHONPATH}"
log "Launching: open ${OPEN_FLAGS[@]} --args ${KLAYOUT_ARGS[@]} ${QT_ARGS[@]} [user args]"

#--------------------------------------------------------------------------
# Execution
#--------------------------------------------------------------------------
command -v open >/dev/null || { echo "'open' not found"; exit 5; }
open "${OPEN_FLAGS[@]}" --args "${KLAYOUT_ARGS[@]}" "${QT_ARGS[@]}" "$@"
