
/*
  KLayout Layout Viewer
  Copyright (C) 2006-2019 Matthias Koefferlein
*/

#include "dbMEBESWriter.h"
#include "dbPolygonTools.h"
#include "tlException.h"
#include "tlInternational.h"
#include "tlLog.h"
#include "tlTimer.h"
#include "tlFileUtils.h"

//  don't compress below a threshold of 3 shapes per array
const size_t array_threshold = 3;

namespace db
{

// ----------------------------------------------------------------------------------

class TrapezoidReducer
  : public db::SimplePolygonSink
{
public:
  TrapezoidReducer (std::unordered_map<db::SimplePolygon, std::vector<db::Vector> > *map)
    : mp_map (map)
  {
    //  .. nothing yet ..
  }

  void put (const db::SimplePolygon &poly)
  {
    if (poly.hull ().size () < 3) {
      return;
    }

    db::Box box = poly.box ();
    if (box.height () > 1024) {

      if (poly.is_box ()) {

        db::Coord y = box.bottom ();
        while (y < box.top ()) {
          put (db::SimplePolygon (db::Box (box.left (), y, box.right (), std::min (y + 1024, box.top ()))));
          y += 1024;
        }

      } else {

        std::vector <db::SimplePolygon> clip_result;

        db::Coord y = box.bottom ();
        while (y < box.top ()) {

          db::Box clip_box (box.left (), y, box.right (), std::min (y + 1024, box.top ()));

          clip_result.clear ();
          db::clip_poly (poly, clip_box, clip_result);
          for (std::vector <db::SimplePolygon>::const_iterator q = clip_result.begin (); q != clip_result.end (); ++q) {
            put (*q);
          }

          y += 1024;

        }

      }

    } else {

      db::Vector pos = box.p1 () - db::Point ();
      db::SimplePolygon p = poly.moved (-pos);

      std::unordered_map<db::SimplePolygon, std::vector<db::Vector> >::iterator im = mp_map->find (p);
      if (im != mp_map->end ()) {
        im->second.push_back (pos);
      } else {
        mp_map->insert (std::make_pair (p, std::vector<db::Vector> ())).first->second.push_back (pos);
      }

    }
  }

public:
  std::unordered_map<db::SimplePolygon, std::vector<db::Vector> > *mp_map;
};

// ----------------------------------------------------------------------------------

struct SubresTransformation
{
public:
  SubresTransformation ()
  {
    //  nothing yet ..
  }

  db::Vector operator() (const db::Vector &other) const
  {
    return db::Vector (other.x () * 16, other.y ());
  }

  db::Point operator() (const db::Point &other) const
  {
    return db::Point (other.x () * 16, other.y ());
  }

  bool is_ortho () const
  {
    return true;
  }
};

db::Point operator* (const SubresTransformation &t, const db::Point &p)
{
  return t (p);
}

// ----------------------------------------------------------------------------------

class DecompositionCache
{
public:
  typedef std::unordered_map<db::SimplePolygon, std::vector<db::Vector> > norm_results;

  DecompositionCache (norm_results *results, bool cache_enabled, db::Coord smooth_dist, const db::ICplxTrans &final_trans, bool subres)
    : mp_results (results), m_cache_enabled (cache_enabled), m_smooth_dist (smooth_dist), m_final_trans (final_trans), m_subres (subres)
  { }

  /**
   *  @brief Decompose the given polygon using caching if required
   *
   *  Note that the argument is a reference by intention. This way, the argument's polygon can
   *  be modified internally without creating a copy.
   */
  void decompose(db::Polygon &poly)
  {
    if (poly.is_box ()) {

      poly.transform (m_final_trans);
      if (m_subres) {
        poly.transform (SubresTransformation ());
      }

      TrapezoidReducer reducer (mp_results);
      db::decompose_trapezoids (poly, TD_simple, reducer);

    } else if (m_cache_enabled) {

      db::Vector offset = poly.box ().p1 () - db::Point ();
      poly.move (-offset);

      std::map<db::Polygon, norm_results>::iterator c = m_cache.find (poly);
      if (c == m_cache.end ()) {

        c = m_cache.insert (std::make_pair (poly, norm_results ())).first;

        if (m_smooth_dist) {
#if defined(KLAYOUT_SMOOTH_HAS_KEEP_HV)
          poly = db::smooth (poly, m_smooth_dist, true);
#else
          poly = db::smooth (poly, m_smooth_dist);
#endif
        }
        poly.transform (m_final_trans);
        if (m_subres) {
          poly.transform (SubresTransformation ());
        }

        TrapezoidReducer reducer (&c->second);
        db::decompose_trapezoids (poly, TD_simple, reducer);

      }

      offset = m_final_trans * offset;
      if (m_subres) {
        offset = SubresTransformation () (offset);
      }

      for (norm_results::iterator r = c->second.begin (); r != c->second.end (); ++r) {
        std::vector<db::Vector> &disp = (*mp_results)[r->first];
        std::vector<db::Vector> new_disp = r->second;
        for (std::vector<db::Vector>::iterator d = new_disp.begin (); d != new_disp.end (); ++d) {
          *d += offset;
        }
        disp.insert (disp.end (), new_disp.begin (), new_disp.end ());
      }

    } else {

      if (m_smooth_dist) {
#if defined(KLAYOUT_SMOOTH_HAS_KEEP_HV)
        poly = db::smooth (poly, m_smooth_dist, true);
#else
        poly = db::smooth (poly, m_smooth_dist);
#endif
      }
      poly.transform (m_final_trans);
      if (m_subres) {
        poly.transform (SubresTransformation ());
      }

      TrapezoidReducer reducer (mp_results);
      db::decompose_trapezoids (poly, TD_simple, reducer);

    }
  }

private:
  std::map<db::Polygon, norm_results> m_cache;
  norm_results *mp_results;
  bool m_cache_enabled;
  db::Coord m_smooth_dist;
  db::ICplxTrans m_final_trans;
  bool m_subres;
};

// ----------------------------------------------------------------------------------

MEBESWriter::MEBESWriter (const std::string &path)
  : m_next_segment (0), m_next_stripe (0),
    m_segments (0), m_stripes (0), m_cluster_size (1),
    m_timestamp_enabled (true),
    m_fracture_caching_enabled (false),
    m_subresolution_fracturing_enabled (false),
    m_smoothing_enabled (false),
    m_address_unit (0.0), m_compression_level (2),
    m_max_storage_size (0), m_timer (tl::verbosity () > 20, "MEBES writer"),
    m_path (path), m_directory_pos (0), m_has_errors (false)
{
  //  .. nothing yet ..
}

MEBESWriter::~MEBESWriter ()
{
  //  .. nothing yet ..
}

void MEBESWriter::enable_timestamp (bool en)
{
  m_timestamp_enabled = en;
}

void MEBESWriter::enable_smoothing (bool en)
{
  m_smoothing_enabled = en;
}

void MEBESWriter::enable_subresolution_fracturing (bool en)
{
  m_subresolution_fracturing_enabled = en;
}

void MEBESWriter::enable_fracture_caching (bool en)
{
  m_fracture_caching_enabled = en;
}

void MEBESWriter::set_pattern_name (const std::string &p)
{
  m_pattern_name = p;
}

void MEBESWriter::set_maskshop_info (const std::string &s)
{
  m_maskshop_info = s;
}

void MEBESWriter::set_compression_level (unsigned int l)
{
  m_compression_level = l;
}

void MEBESWriter::set_address_unit (double unit)
{
  m_address_unit = unit;
}

void MEBESWriter::set_data (int n, const std::string &s)
{
  m_data[n] = s;
}

const std::string &MEBESWriter::data (int n) const
{
  std::map<int, std::string>::const_iterator i = m_data.find (n);
  if (i != m_data.end ()) {
    return i->second;
  } else {
    static const std::string s_empty;
    return s_empty;
  }
}

void MEBESWriter::begin (size_t nx, size_t ny, const db::DPoint &p0, double dx, double dy, const db::DBox &frame)
{
  mp_stream.reset (new tl::OutputStream (m_path));

  //  suppress some diagnostics:
  m_has_errors = true;

  if (m_address_unit < 1e-10) {
    throw tl::Exception (tl::to_string (tr ("Address unit not configured or zero")));
  }

  int sw = int (floor (0.5 + dx / m_address_unit + 1e-10));
  int sh = int (floor (0.5 + dy / m_address_unit + 1e-10));

  if (sw > 32768) {
    throw tl::Exception (tl::to_string (tr ("Tile width too large for MEBES format - must be 32768 address units wide")));
  } else if (sw < 32768) {
    throw tl::Exception (tl::to_string (tr ("Tile width too small for MEBES format - must be 32768 address units wide")));
  }

  if (sh > 65536) {
    throw tl::Exception (tl::to_string (tr ("Tile height too large for MEBES format - must be 65536 address units max")));
  } else if (sh < 256) {
    throw tl::Exception (tl::to_string (tr ("Tile height too small for MEBES format - must be 256 address units min")));
  } else if (((sh - 1) & sh) != 0) {
    throw tl::Exception (tl::to_string (tr ("Tile height not compatible with MEBES format - must be a power-of-two multiple of the address unit")));
  }

  m_offset = p0;
  m_segments = nx;
  m_cluster_size = (sh > 1024 ? sh / 1024 : 1);

  if (! mp_stream->supports_seek ()) {
    throw tl::Exception (tl::to_string (tr ("Cannot write: stream does not support seek operations")));
  }

  m_frame = frame;
  if (! m_frame.p1 ().equal (m_offset)) {
    throw tl::Exception (tl::to_string (tr ("Tile origin is not configured to match the frame's lower left corner")));
  }

  db::Box frame_in_au = db::Box (m_frame * (1.0 / m_address_unit));

  if (m_cluster_size == 1) {
    m_stripes = ny;
  } else {
    m_stripes = (frame_in_au.height () + 1024 - 1) / 1024;
  }

  if (tl::verbosity () >= 20) {
    tl::log << "MEBES writer parameters:";
    tl::log << "  Segments = " << m_segments;
    tl::log << "  Stripes = " << m_stripes;
    tl::log << "  Stripe height = " << (sh / m_cluster_size);
    tl::log << "  Stripe cluster = " << m_cluster_size;
  }

  // write header
  int num_data_fields = 3;
  if (! m_data.empty ()) {
    num_data_fields += (--m_data.end ())->first + 1;
  }
  write_word ((num_data_fields << 8) + 1);  //  MODE 5

  //  address unit
  write_dword ((unsigned int) floor (0.5 + 1e-10 + m_address_unit * double (1 << 28)));

  //  stripe height
  write_word ((unsigned int) (sh / m_cluster_size));

  //  chip width and height
  write_dword (frame_in_au.width ());
  write_dword (frame_in_au.height ());

  //  month, day, year
  if (m_timestamp_enabled) {

    time_t now = time (NULL);
    struct tm tm_struct;
#if defined(_MSC_VER) || defined(__MINGW32__)
    localtime_s (&tm_struct, &now);
#else
    localtime_r (&now, &tm_struct);
#endif

    write_word (tm_struct.tm_mon + 1);
    write_word (tm_struct.tm_mday);
    write_word (tm_struct.tm_year + 1900);

  } else {
    write_word (1);
    write_word (1);
    write_word (0);
  }

  //  data field #1
  write_word (6);

  std::string pn = tl::to_upper_case (m_pattern_name);
  std::string pn_red;
  for (size_t i = 0; i < pn.size () && i < 11; ++i) {
    if (isalnum (pn [i])) {
      pn_red += pn [i];
    }
  }
  while (pn_red.size () < 11) {
    pn_red += 'X';
  }
  mp_stream->put ((std::string (pn_red, 0, 9) + "." + std::string (pn_red, 9, 2)).c_str (), 12);

  //  data field #2
  write_dword ((unsigned int) (m_segments * 2 + 2));

  //  placeholder for directory to be written later
  m_directory_pos = mp_stream->pos ();
  m_directory.resize (m_segments, 0);
  for (size_t i = 0; i <= m_segments; ++i) {
    write_dword (0);
  }

  //  data field #3
  size_t maskshop_info_size = 40;
  std::string mi = m_maskshop_info;
  while (mi.size () < maskshop_info_size) {
    mi += ' ';
  }
  write_word ((unsigned int) (maskshop_info_size / 2));
  mp_stream->put (mi.c_str (), maskshop_info_size);

  //  additional data fields
  if (! m_data.empty ()) {

    int nto = (--m_data.end ())->first;
    for (int n = 0; n <= nto; ++n) {

      std::string s;
      if (m_data.find (n) != m_data.end ()) {
        s = m_data[n];
      }

      s += '\0';

      //  data field #1
      size_t sz = (s.size () / 2) * 2;
      write_word ((unsigned int) (sz / 2));
      mp_stream->put (s.c_str (), sz);

    }

  }

  //  end of header
  finish_record ();

  before_first_segment ();

  m_has_errors = false;
}

void MEBESWriter::write_word (unsigned int w)
{
  char b[2] = { (char) (w >> 8), (char) w };
  mp_stream->put (b, 2);
}

void MEBESWriter::write_dword (unsigned int w)
{
  char b[4] = { (char) (w >> 24), (char) (w >> 16), (char) (w >> 8), (char) w };
  mp_stream->put (b, 4);
}

void MEBESWriter::finish_record ()
{
  size_t p = mp_stream->pos ();
  if ((p & 2047) != 0) {
    write_word (9);
    p += 2;
  }
  while ((p & 2047) != 0) {
    write_word (0);
    p += 2;
  }
}

void MEBESWriter::require_bytes (size_t n)
{
  size_t p = mp_stream->pos ();
  if (((p + n) & ~size_t (2047)) != (p & ~size_t (2047))) {
    finish_record ();
  }
}

void MEBESWriter::write_array (const TrapezoidArraySpec &ta)
{
  write_word (21);

  if (m_subresolution_fracturing_enabled) {

    tl_assert (ta.dx >= 0 && ta.dx < 32768 * 16);
    tl_assert ((ta.dx % 16) == 0);
    tl_assert (ta.dy >= 0 && ta.dy < 65536);
    tl_assert (ta.nx > 0 && ta.nx < 32768);
    tl_assert (ta.ny > 0 && ta.ny < 65536);

    write_word (ta.dx / 16);
    write_word (ta.dy);
    write_word ((unsigned int) ta.nx);
    write_word ((unsigned int) ta.ny);

  } else {

    tl_assert (ta.dx >= 0 && ta.dx < 32768);
    tl_assert (ta.dy >= 0 && ta.dy < 65536);
    tl_assert (ta.nx > 0 && ta.nx < 32768);
    tl_assert (ta.ny > 0 && ta.ny < 65536);

    write_word (ta.dx);
    write_word (ta.dy);
    write_word ((unsigned int) ta.nx);
    write_word ((unsigned int) ta.ny);

  }
}

void MEBESWriter::write_figure (const db::SimplePolygon &poly, const TrapezoidArraySpec &ta)
{
  db::Point p1, p2, p3, p4;

  if (poly.hull ().size () == 4) {

    p1 = poly.hull ()[0];
    p2 = poly.hull ()[1];
    p3 = poly.hull ()[2];
    p4 = poly.hull ()[3];

  } else {

    tl_assert (poly.hull ().size () == 3);

    db::Point pa = poly.hull ()[0];
    db::Point pb = poly.hull ()[1];
    db::Point pc = poly.hull ()[2];

    if (pa.y () == pc.y ()) {

      p1 = pa;
      p2 = pb;
      p3 = pb;
      p4 = pc;

    } else if (pb.y () == pc.y ()) {

      p1 = pa;
      p2 = pb;
      p3 = pc;
      p4 = pa;

    } else {
      tl_assert (false);
    }

  }

  db::Point p0 = p1 + ta.offset;

  tl_assert (p1.y () == p4.y ());
  tl_assert (p2.y () == p3.y ());
  tl_assert (p2.y () > p1.y ());
  tl_assert (p3.x () >= p2.x ());
  tl_assert (p4.x () >= p1.x ());
  tl_assert (p0.x () >= 0 && p0.x () <= (m_subresolution_fracturing_enabled ? 16 * 32768 : 32768));
  tl_assert (p0.y () >= 0 && p0.y () < 1024);

  unsigned int h = p2.y () - p1.y ();
  unsigned int w = p4.x () - p1.x ();
  tl_assert (h <= 1024);

  size_t array_bytes = 0;

  //  NOTE: if the figure extends beyond the stripe boundary we write a single-item
  //  array to avoid splitting. MEBES allows for arrays to cross the stripe boundary
  //  but not single figures
  if (ta.nx > 1 || ta.ny > 1 || p0.y () + h > 1024) {
    array_bytes = 10;
  }

  if (p1.x () == p2.x () && p3.x () == p4.x () && (!m_subresolution_fracturing_enabled || ((p1.x () % 16) == 0 && (p3.x () % 16) == 0))) {

    //  Rectangle
    require_bytes (8 + array_bytes);
    if (array_bytes > 0) {
      write_array (ta);
    }

    write_word (((h - 1) << 6) + 16);

    if (m_subresolution_fracturing_enabled) {

      write_word (w / 16);
      write_word (p0.x () / 16);
      write_word (p0.y ());

    } else {

      write_word (w);
      write_word (p0.x ());
      write_word (p0.y ());

    }

  } else if (p2.x () - p1.x () == p3.x () - p4.x () || p2.x () == p1.x () || p3.x () == p4.x ()) {

    unsigned int type = 17;
    if (p2.x () == p1.x ()) {
      type = 19;
    } else if (p3.x () == p4.x ()) {
      type = 18;
    }

    //  Parallelogram or left-side-vertical/right-side-vertical trapezoid
    require_bytes (12 + array_bytes);
    if (array_bytes > 0) {
      write_array (ta);
    }

    int dx = (type == 19) ? (p3.x () - p4.x ()) : (p2.x () - p1.x ());

    write_word (((h - 1) << 6) + type);

    if (m_subresolution_fracturing_enabled) {

      write_word (w & 0xffff);
      write_word (p0.x () & 0xffff);
      write_word (p0.y ());
      write_word (dx & 0xffff);
      write_word (((unsigned int) (dx >> 6) & (unsigned int) 0xfc00) |
                  ((unsigned int) (p0.x () >> 11) & (unsigned int) 0x03e0) |
                  ((unsigned int) (w >> 16) & (unsigned int) 0x001f));

    } else {

      write_word (w << 4);
      write_word (p0.x () << 4);
      write_word (p0.y ());
      write_word (dx << 4);
      write_word (((unsigned int) (dx >> 2) & (unsigned int) 0xfc00) |
                  ((unsigned int) (p0.x () >> 7) & (unsigned int) 0x03e0) |
                  ((unsigned int) (w >> 12) & (unsigned int) 0x001f));

    }

  } else {

    //  Generic trapezoid
    require_bytes (14 + array_bytes);
    if (array_bytes > 0) {
      write_array (ta);
    }

    int dx1 = p2.x () - p1.x ();
    int dx2 = p3.x () - p4.x ();
    tl_assert (p0.y () < 1024);

    write_word (((h - 1) << 6) + 20);

    if (m_subresolution_fracturing_enabled) {

      write_word (w & 0xffff);
      write_word (p0.x () & 0xffff);
      write_word (((unsigned int) p0.y () & 0x3ff) | ((unsigned int) (dx2 >> 6) & 0xfc00));
      write_word (dx1 & 0xffff);
      write_word (dx2 & 0xffff);
      write_word (((unsigned int) (dx1 >> 6) & (unsigned int) 0xfc00) |
                  ((unsigned int) (p0.x () >> 11) & (unsigned int) 0x03e0) |
                  ((unsigned int) (w >> 16) & (unsigned int) 0x001f));

    } else {

      write_word (w << 4);
      write_word (p0.x () << 4);
      write_word (((unsigned int) p0.y () & 0x3ff) | ((unsigned int) (dx2 >> 2) & 0xfc00));
      write_word (dx1 << 4);
      write_word (dx2 << 4);
      write_word (((unsigned int) (dx1 >> 2) & (unsigned int) 0xfc00) |
                  ((unsigned int) (p0.x () >> 7) & (unsigned int) 0x03e0) |
                  ((unsigned int) (w >> 12) & (unsigned int) 0x001f));

    }

  }
}

void MEBESWriter::before_first_segment ()
{
  m_needs_new_segment = false;
}

void MEBESWriter::before_stripe ()
{
  m_start_written = false;
  if (m_next_stripe == 0) {
    m_needs_new_segment = true;
  }
}

void MEBESWriter::before_figure ()
{
  if (m_start_written) {
    return;
  }

  //  NOTE: we must write a segment start entry, hence we write even if we optimize out
  if (m_needs_new_segment) {

    finish_record ();

    tl_assert (m_next_segment < m_directory.size ());
    m_directory[m_next_segment] = (unsigned int) (mp_stream->pos () / size_t (2048));

    //  segment start
    write_word (12);
    write_dword ((unsigned int) (m_next_segment + 1));

    m_needs_new_segment = false;

  }

  //  stripe start
  require_bytes (6);
  write_word (13);
  write_dword ((unsigned int) (m_next_stripe + 1));

  m_start_written = true;
}

void MEBESWriter::after_stripe ()
{
  //  end of stripe
  if (m_start_written) {
    require_bytes (2);
    write_word (8);
  }

  ++m_next_stripe;
  if (m_next_stripe == m_stripes) {
    m_next_stripe = 0;
    ++m_next_segment;
  }
}

FracturedData MEBESWriter::fracture (const db::Region &region, const db::Box &org_tile, double dbu) const
{
  FracturedData result;

  std::vector <db::Polygon> clip_result;

  db::Box frame = db::Box (m_frame * (1.0 / dbu)) & org_tile;

  db::ICplxTrans final_trans = db::ICplxTrans (dbu / m_address_unit) * db::ICplxTrans (db::Vector (-org_tile.left (), -org_tile.bottom ()));

  DecompositionCache decomposition (&result.normalized, m_fracture_caching_enabled, m_smoothing_enabled ? db::Coord (m_address_unit * 0.5 / dbu) : 0, final_trans, m_subresolution_fracturing_enabled);

  for (db::Region::const_iterator p = region.begin_merged (); !p.at_end (); ++p) {

    if (frame.empty ()) {
      //  clipped away
    } else if (!p->box ().inside (frame)) {

      clip_result.clear ();
      db::clip_poly (*p, frame, clip_result);
      for (std::vector <db::Polygon>::iterator q = clip_result.begin (); q != clip_result.end (); ++q) {
        decomposition.decompose (*q);
      }

    } else {
      db::Polygon sp = *p;
      decomposition.decompose (sp);
    }

  }

  //  produce the repetitions

  typedef db::DisplacementCompressor::disp_vector disp_vector;
  typedef db::DisplacementCompressor::array_vector array_vector;
  std::vector<std::pair<db::SimplePolygon, array_vector> > arrays;

  for (std::unordered_map<db::SimplePolygon, disp_vector>::iterator n = result.normalized.begin (); n != result.normalized.end (); ++n) {

    if (m_compression_level >= 1 && n->second.size () >= array_threshold) {

      db::DisplacementCompressor compressor;

      compressor.swap_displacements (n->second);
      compressor.compress (m_compression_level, m_subresolution_fracturing_enabled);
      compressor.swap_displacements (n->second);

      result.arrays.push_back (std::make_pair (n->first, array_vector ()));
      compressor.swap_arrays (result.arrays.back ().second);

    }

  }

  return result;
}

void MEBESWriter::deliver_tile (const db::FracturedData &td)
{
  if (m_cluster_size > 1) {

    size_t current_segment = m_next_segment;
    for (size_t i = 0; i < m_cluster_size && m_next_segment == current_segment; ++i) {

      db::Coord y0 = db::Coord (i * 1024);
      db::Coord y1 = db::Coord ((i + 1) * 1024);

      before_stripe ();

      //  TODO: drop the start/finish/record if there is nothing written

      for (std::vector<std::pair<db::SimplePolygon, db::FracturedData::array_vector> >::const_iterator a = td.arrays.begin (); a != td.arrays.end (); ++a) {

        for (std::vector<TrapezoidArraySpec>::const_iterator r = a->second.begin (); r != a->second.end (); ++r) {

          if (r->offset.y () >= y0 && r->offset.y () < y1) {

            before_figure ();

            TrapezoidArraySpec ta = *r;
            ta.offset -= db::Vector (0, y0);
            write_figure (a->first, ta);

          }

        }

      }

      TrapezoidArraySpec ta_single;
      for (std::unordered_map <db::SimplePolygon, db::FracturedData::disp_vector>::const_iterator n = td.normalized.begin (); n != td.normalized.end (); ++n) {

        for (db::FracturedData::disp_vector::const_iterator d = n->second.begin (); d != n->second.end (); ++d) {

          if (d->y () >= y0 && d->y () < y1) {

            before_figure ();

            ta_single.offset = db::Vector (d->x (), d->y () - y0);
            write_figure (n->first, ta_single);

          }

        }

      }

      after_stripe ();

    }

  } else {

    before_stripe ();

    for (std::vector<std::pair<db::SimplePolygon, db::FracturedData::array_vector> >::const_iterator a = td.arrays.begin (); a != td.arrays.end (); ++a) {

      for (std::vector<TrapezoidArraySpec>::const_iterator r = a->second.begin (); r != a->second.end (); ++r) {

        before_figure ();
        write_figure (a->first, *r);

      }

    }

    TrapezoidArraySpec ta_single;
    for (std::unordered_map <db::SimplePolygon, db::FracturedData::disp_vector>::const_iterator n = td.normalized.begin (); n != td.normalized.end (); ++n) {

      for (db::FracturedData::disp_vector::const_iterator d = n->second.begin (); d != n->second.end (); ++d) {

        before_figure ();

        ta_single.offset = db::Vector (*d);
        write_figure (n->first, ta_single);

      }

    }

    after_stripe ();

  }
}

void MEBESWriter::deliver_stored ()
{
  std::map<std::pair<size_t, size_t>, FracturedData>::iterator im;
  while ((im = m_tile_storage.find (std::make_pair (m_next_segment, m_next_stripe))) != m_tile_storage.end ()) {

    deliver_tile (im->second);
    m_tile_storage.erase (im);

  }
}

void MEBESWriter::put (size_t ix, size_t iy, const db::Box &tile, size_t /*id*/, const tl::Variant &obj, double dbu, const db::ICplxTrans & /*trans*/, bool /*clip*/)
{
  try {

    if (! m_has_errors) {
      tl_assert (m_next_segment < m_segments && m_next_stripe < m_stripes);
    }
    deliver_stored ();

    FracturedData fractured;

    if (obj.is_user<db::Region> ()) {

      const db::Region &region = obj.to_user<db::Region> ();
      fractured = fracture (region, tile, dbu);

    } else if (obj.is_user<FracturedData> ()) {

      fractured = obj.to_user<FracturedData> ();

    }

    if (ix == m_next_segment && iy * m_cluster_size == m_next_stripe) {

      deliver_tile (fractured);

    } else {

      m_tile_storage.insert (std::make_pair (std::make_pair (ix, iy * m_cluster_size), FracturedData ())).first->second.swap (fractured);

      if (m_tile_storage.size () > m_max_storage_size) {
        m_max_storage_size = m_tile_storage.size ();
      }

    }

  } catch (...) {
    m_has_errors = true;
    throw;
  }
}

void MEBESWriter::finish (bool success)
{
  if (! success || m_has_errors) {

    m_has_errors = true;

    m_tile_storage.clear ();
    m_directory.clear ();
    mp_stream.reset (0);

    tl::rm_file (m_path);

    tl::log << "MEBES file deleted because of errors: " << m_path;

  } else {

    deliver_stored ();
    tl_assert (m_next_segment == m_segments && m_next_stripe == 0);

    //  end of drawing
    require_bytes (2);
    write_word (4);

    finish_record ();

    unsigned int length = (unsigned int) (mp_stream->pos () / size_t (2048));

    //  write directory
    mp_stream->seek (m_directory_pos);
    write_dword (length);
    for (std::vector<unsigned int>::const_iterator d = m_directory.begin (); d != m_directory.end (); ++d) {
      write_dword (*d > 0 ? *d + 1 : 0);
    }

    if (tl::verbosity () >= 20) {
      tl::log << "MEBES writer tile storage max size = " << m_max_storage_size;
    }
    tl::log << "MEBES file written: " << m_path;

    m_tile_storage.clear ();
    m_directory.clear ();

    mp_stream.reset (0);

  }
}

}
