
/*
  KLayout Layout Viewer
  Copyright (C) 2006-2019 Matthias Koefferlein
*/

#include "dbMEBESCompression.h"
#include "dbHash.h"

namespace db
{

// ----------------------------------------------------------------
//  TrapezoidArraySpec implementation

TrapezoidArraySpec::TrapezoidArraySpec ()
  : offset (), nx (1), ny (1), dx (0), dy (0)
{
  //  .. nothing yet ..
}

TrapezoidArraySpec::TrapezoidArraySpec (const db::Vector &_offset, size_t _nx, size_t _ny, db::Coord _dx, db::Coord _dy)
  : offset (_offset), nx (_nx), ny (_ny), dx (_nx == 1 ? 0 : _dx), dy (_ny == 1 ? 0 : _dy)
{
  //  .. nothing yet ..
}

// ----------------------------------------------------------------
//  DisplacementCompressor implementation

namespace {

//  TODO: this is copy & pasted from OASISWriter

/**
 *  @brief Compare operator for points, distinct x clustered (with same y)
 */
struct vector_cmp_x
{
  bool operator() (const db::Vector &a, const db::Vector &b) const
  {
    if (a.y () != b.y ()) {
      return a.y () < b.y ();
    } else {
      return a.x () < b.x ();
    }
  }
};

/**
 *  @brief Compare operator for points, distinct y clustered (with same x)
 */
struct vector_cmp_y
{
  bool operator() (const db::Vector &a, const db::Vector &b) const
  {
    if (a.x () != b.x ()) {
      return a.x () < b.x ();
    } else {
      return a.y () < b.y ();
    }
  }
};

/**
 *  @brief Compare operator for points/abstract repetition pair with configurable point compare operator
 */
template <class PC>
struct rep_vector_cmp
{
  bool operator () (const std::pair <db::Vector, std::pair <db::Coord, int> > &a, const std::pair <db::Vector, std::pair <db::Coord, int> > &b)
  {
    if (a.second.first != b.second.first) {
      return a.second.first < b.second.first;
    }
    PC pc;
    return pc (a.first, b.first);
  }
};

template <class R>
inline R safe_diff (R a, R b)
{
  R d = a - b;
  if ((a > b && d < 0) || (a < b && d > 0)) {
    throw tl::Exception ("Signed coordinate difference overflow");
  }
  return d;
}

}

DisplacementCompressor::DisplacementCompressor ()
{
  //  .. nothing yet ..
}

void DisplacementCompressor::clear ()
{
  m_displacements.clear ();
  m_arrays.clear ();
}

void DisplacementCompressor::compress (unsigned int compression_level, bool subresolution_fracturing_enabled)
{
  bool array_subtraction = ((compression_level / 100) == 1);
  compression_level %= 100;

  disp_vector::iterator d;

  std::unordered_set<db::Coord> xcoords, ycoords;
  if (compression_level > 1) {
    for (d = m_displacements.begin (); d != m_displacements.end (); ++d) {
      xcoords.insert (d->x ());
      ycoords.insert (d->y ());
    }
  }

  bool xfirst = xcoords.size () < ycoords.size ();

  disp_vector new_displacements;
  typedef std::vector <std::pair <db::Vector, std::pair <db::Coord, int> > > tmp_rep_vector;
  tmp_rep_vector new_arrays;

  m_arrays.clear ();

  //  Try single-point compression to repetitions in the x and y direction. For the first
  //  direction, use the one with more distinct values. For this, a better compression is
  //  expected.
  for (int xypass = 0; xypass <= 1; ++xypass) {

    new_arrays.clear ();

    bool xrep = (xfirst == (xypass == 0));

    if (xrep) {
      std::sort (m_displacements.begin (), m_displacements.end (), vector_cmp_x ());
    } else {
      std::sort (m_displacements.begin (), m_displacements.end (), vector_cmp_y ());
    }

#if 0
    //  maybe this is required?
    disp_vector::iterator e = std::unique (m_displacements.begin (), m_displacements.end ());
    m_displacements.erase (e, m_displacements.end ());
#endif

    disp_vector::iterator dwindow = m_displacements.begin ();
    for (d = m_displacements.begin (); d != m_displacements.end (); ) {

      //  collect the nearest neighbor distances and counts for 2..level order neighbors
      int nxy_max = 1;
      unsigned int nn_max = 0;

      //  move the window of identical x/y coordinates if necessary
      if (d == dwindow) {
        for (dwindow = d + 1; dwindow != m_displacements.end () && (xrep ? (dwindow->y () == d->y ()) : (dwindow->x () == d->x ())); ++dwindow)
          ;
      }

      for (unsigned int nn = 0; nn < compression_level; ++nn) {

        disp_vector::iterator dd = d + (nn + 1);
        if (dd >= dwindow) {
          break;
        }

        db::Vector dxy = xrep ? db::Vector (safe_diff (dd->x (), d->x ()), 0) : db::Vector (0, safe_diff (dd->y (), d->y ()));
        if (subresolution_fracturing_enabled && (dxy.x () % 16) != 0) {
          //  In subresolution mode, the arrays need to be spaced by multiples of 16 units.
          //  This is usually fulfilled, but for safety we ensure it is.
          continue;
        }

        int nxy = 2;
        while (dd != dwindow) {
          disp_vector::iterator df = std::lower_bound (dd + 1, dwindow, *dd + dxy);
          if (df == dwindow || *df != *dd + dxy) {
            break;
          }
          ++nxy;
          dd = df;
        }

        if (nxy > nxy_max) {
          nxy_max = nxy;
          nn_max = nn;
        }

      }

      if (nxy_max < 2) {

        //  no candidate found - just keep that one
        new_displacements.push_back (*d++);

      } else {

        //  take out the ones of this sequence from the list
        db::Vector dxy_max = xrep ? db::Vector (safe_diff ((d + nn_max + 1)->x (), d->x ()), 0) : db::Vector (0, safe_diff ((d + nn_max + 1)->y (), d->y ()));

        disp_vector::iterator ds = dwindow;
        disp_vector::iterator dt = dwindow;
        db::Vector df = *d + dxy_max * long (nxy_max - 1);

        do {
          --ds;
          if (*ds != df) {
            *--dt = *ds;
          } else {
            df -= dxy_max;
          }
        } while (ds != d);

        new_arrays.push_back (std::make_pair (*d, std::make_pair (xrep ? dxy_max.x () : dxy_max.y (), nxy_max)));

        d = dt;

      }

    }

    m_displacements.clear ();
    m_displacements.swap (new_displacements);

    tmp_rep_vector carry;

    do {

      //  Try to compact these repetitions further in the perpendicular direction
      if (xrep) {
        std::sort (new_arrays.begin (), new_arrays.end (), rep_vector_cmp<vector_cmp_y> ());
      } else {
        std::sort (new_arrays.begin (), new_arrays.end (), rep_vector_cmp<vector_cmp_x> ());
      }

      tmp_rep_vector::iterator rw = new_arrays.begin ();

      for (tmp_rep_vector::iterator r = new_arrays.begin (); r != new_arrays.end (); ) {

        tmp_rep_vector::iterator rf = r;
        db::Coord pos = r->second.first;
        while (rf != new_arrays.end () && rf->second.first == pos) {
          ++rf;
        }

        while (r != rf) {

          //  within the identical repetitions determine the displacement with the maximum
          //  potential for compression in perpendicular direction.

          int nxy2_max = 1;
          int nxy1_max = r->second.second;
          int ntot_max = r->second.second;
          db::Vector dxy2_max;

          for (unsigned int nn2 = 0; nn2 < compression_level; ++nn2) {

            tmp_rep_vector::iterator rn = r + (nn2 + 1);
            if (rn >= rf) {
              break;
            }

            db::Vector dxy2 = xrep ? db::Vector (0, safe_diff (rn->first.y (), r->first.y ())) : db::Vector (safe_diff (rn->first.x (), r->first.x ()), 0);
            if (subresolution_fracturing_enabled && (dxy2.x () % 16) != 0) {
              //  In subresolution mode, the arrays need to be spaced by multiples of 16 units.
              //  This is usually fulfilled, but for safety we ensure it is.
              continue;
            }

            int nxy2 = 1;
            int ntot = r->second.second;
            int nxy1 = r->second.second;
            int nxy1_min = nxy1;

            int n = 0;
            db::Vector dxy2n;
            for (tmp_rep_vector::iterator rr = r; rr != rf; ++rr) {

              if (rr->first == r->first + dxy2n && (array_subtraction || rr->second.second == nxy1)) {

                ++n;
                dxy2n += dxy2;

                nxy1_min = std::min (nxy1_min, rr->second.second);
                int nt = nxy1_min * n;
                if (nt > ntot) {
                  nxy2 = n;
                  nxy1 = nxy1_min;
                  ntot = nt;
                }

              }

            }

            if (ntot > ntot_max) {
              ntot_max = ntot;
              nxy1_max = nxy1;
              nxy2_max = nxy2;
              dxy2_max = dxy2;
            }

          }

          if (nxy1_max == 1 && nxy2_max == 1) {

            m_displacements.push_back (r->first);

          } else {

            TrapezoidArraySpec ta;
            if (xrep) {
              ta = TrapezoidArraySpec (r->first, nxy1_max, nxy2_max, r->second.first, dxy2_max.y ());
            } else {
              ta = TrapezoidArraySpec (r->first, nxy2_max, nxy1_max, dxy2_max.x (), r->second.first);
            }
            m_arrays.push_back (ta);

          }

          carry.clear ();

          {
            int n = nxy2_max;
            db::Vector dn = r->first;
            for (tmp_rep_vector::iterator rr = r; rr != rf; ++rr) {

              if (n > 0 && rr->first == dn && (array_subtraction || rr->second.second == nxy1_max)) {

                --n;
                dn += dxy2_max;

                if (rr->second.second != nxy1_max) {

                  tl_assert (rr->second.second > nxy1_max);

                  if (rw != rr) {
                    *rw = *rr;
                  }

                  if (xrep) {
                    rw->first += db::Vector (nxy1_max * rw->second.first, 0);
                  } else {
                    rw->first += db::Vector (0, nxy1_max * rw->second.first);
                  }
                  rw->second.second -= nxy1_max;

                  ++rw;

                }

              } else {
                carry.push_back (*rr);
              }

            }
          }

          tl_assert (rf - carry.size () > r);

          r = rf - carry.size ();
          tl_assert (r >= rw);

          tmp_rep_vector::iterator j = r;
          for (tmp_rep_vector::const_iterator i = carry.begin (); i != carry.end (); ++i) {
            *j++ = *i;
          }

        }

      }

      new_arrays.erase (rw, new_arrays.end ());

    } while (! new_arrays.empty ());

  }

}

}
