
/*
  KLayout Layout Viewer
  Copyright (C) 2006-2019 Matthias Koefferlein
*/

#include "dbMEBESWriter.h"
#include "dbMEBESReader.h"
#include "dbTilingProcessor.h"
#include "dbTestSupport.h"
#include "tlUnitTest.h"

#include <stdlib.h>

std::string
layout_meta_info_to_string (const db::Layout &layout);

static std::string testdata (const std::string &fn)
{
  return tl::testsrc () + "/src/plugins/mebes/testdata/" + fn;
}

void write_mebes (const std::string &path, double au, const db::Layout &layout, const db::Cell &cell, unsigned int layer, const db::DBox &frame, int cl, bool subres, bool cache)
{
  db::RecursiveShapeIterator iter (layout, cell, layer);

  db::MEBESWriter *mw = new db::MEBESWriter (path);
  mw->set_address_unit (au);
  mw->set_compression_level (cl);
  mw->enable_subresolution_fracturing (subres);
  mw->enable_fracture_caching (cache);

  db::TilingProcessor proc;
  proc.input ("input", iter, db::ICplxTrans ());
  proc.output ("mw", 0, mw, db::ICplxTrans ());
  proc.set_frame (frame);
  proc.set_dbu (au);
  proc.tile_origin (frame.left (), frame.bottom ());
  proc.tile_size (32768 * au, 65536 * au);

  proc.queue ("_output(mw, _rec(mw).fracture(input, _tile.bbox, _dbu))");

  proc.execute ("");
}

void run_test (tl::TestBase *_this, const std::string &lf, const std::string &mf, const std::string &au_file, double dbu, int cl, bool subres, bool cache = false)
{
  db::Layout layout, layout_au;

  {
    std::string fn = testdata (lf);
    tl::InputStream stream (fn);
    db::Reader reader (stream);

    db::LayerMap map = reader.read (layout);
#if KLAYOUT_MAJOR_VERSION > 0 || (KLAYOUT_MAJOR_VERSION == 0 && KLAYOUT_MINOR_VERSION >= 27)
    std::pair<bool, unsigned int> lm = map.first_logical (db::LayerProperties (1, 0));
#else
    std::pair<bool, unsigned int> lm = map.logical (db::LayerProperties (1, 0));
#endif
    tl_assert (lm.first);
    write_mebes (mf, dbu, layout, layout.cell (*layout.begin_top_down ()), lm.second, db::DBox (500, 600, 1500, 1600), cl, subres, cache);

  }

  {
    tl::InputStream stream (mf);
    db::Reader reader (stream);
    reader.read (layout_au);
  }

  EXPECT_EQ (layout_au.dbu (), dbu);

  db::compare_layouts (_this, layout_au, testdata (au_file));
}

TEST(1A)
{
  run_test (_this, "layout1.oas.gz", tmp_file ("TEST1.MB"), "au1a.meb", 0.05, 0, false);
}

TEST(1B)
{
  run_test (_this, "layout1.oas.gz", tmp_file ("TEST1.MB"), "au1b.meb", 0.05, 1, false);
}

TEST(1C)
{
  run_test (_this, "layout1.oas.gz", tmp_file ("TEST1.MB"), "au1c.meb", 0.05, 2, false);
}

TEST(1D)
{
  run_test (_this, "layout1.oas.gz", tmp_file ("TEST1.MB"), "au1d.meb", 0.05, 10, false);
}

TEST(2A)
{
  run_test (_this, "layout1.oas.gz", tmp_file ("TEST2.MB"), "au2a.meb", 0.005, 0, false);
}

TEST(2B)
{
  run_test (_this, "layout1.oas.gz", tmp_file ("TEST2.MB"), "au2b.meb", 0.005, 1, false);
}

TEST(2C)
{
  run_test (_this, "layout1.oas.gz", tmp_file ("TEST2.MB"), "au2c.meb", 0.005, 2, false);
}

TEST(2D)
{
  run_test (_this, "layout1.oas.gz", tmp_file ("TEST2.MB"), "au2d.meb", 0.005, 10, false);
}

TEST(3A)
{
  run_test (_this, "layout1.oas.gz", tmp_file ("TEST3.MB"), "au3a.meb", 0.05, 0, true);
}

TEST(3B)
{
  run_test (_this, "layout1.oas.gz", tmp_file ("TEST3.MB"), "au3b.meb", 0.05, 1, true);
}

TEST(3C)
{
  run_test (_this, "layout1.oas.gz", tmp_file ("TEST3.MB"), "au3c.meb", 0.05, 2, true);
}

TEST(3D)
{
  run_test (_this, "layout1.oas.gz", tmp_file ("TEST3.MB"), "au3d.meb", 0.05, 10, true);
}

TEST(3E)
{
  run_test (_this, "layout1.oas.gz", tmp_file ("TEST3.MB"), "au3e.meb", 0.05, 10, true, true);
}

//  Meta info
TEST(100)
{
  db::Layout layout, layout_au;

  std::string lf = "layout1.oas.gz";
  std::string mf = tmp_file ("TEST100.MB");
  std::string au_file = "au3d.meb";
  int cl = 10;
  double au = 0.05;
  db::DBox frame (500, 600, 1500, 1600);

  {
    std::string fn = testdata (lf);
    tl::InputStream stream (fn);
    db::Reader reader (stream);

    db::LayerMap map = reader.read (layout);
#if KLAYOUT_MAJOR_VERSION > 0 || (KLAYOUT_MAJOR_VERSION == 0 && KLAYOUT_MINOR_VERSION >= 27)
    std::pair<bool, unsigned int> lm = map.first_logical (db::LayerProperties (1, 0));
#else
    std::pair<bool, unsigned int> lm = map.logical (db::LayerProperties (1, 0));
#endif
    tl_assert (lm.first);

    db::RecursiveShapeIterator iter (layout, layout.cell (*layout.begin_top_down ()), lm.second);

    db::MEBESWriter *mw = new db::MEBESWriter (mf);
    mw->set_address_unit (au);
    mw->set_compression_level (cl);
    mw->set_data (0, "mydata0");
    mw->set_data (2, "mydata2x");
    mw->set_pattern_name ("MYPAT");
    mw->set_maskshop_info ("HELLO WORLD");
    mw->enable_timestamp (false);
    mw->enable_fracture_caching (false);
    mw->enable_subresolution_fracturing (true);

    db::TilingProcessor proc;
    proc.input ("input", iter, db::ICplxTrans ());
    proc.output ("mw", 0, mw, db::ICplxTrans ());
    proc.set_frame (frame);
    proc.set_dbu (au);
    proc.tile_origin (frame.left (), frame.bottom ());
    proc.tile_size (32768 * au, 65536 * au);

    proc.queue ("_output(mw, _rec(mw).fracture(input, _tile.bbox, _dbu))");

    proc.execute ("");

  }

  {
    tl::InputStream stream (mf);
    db::Reader reader (stream);
    db::LayerMap map = reader.read (layout_au);

    EXPECT_EQ (layout_au.dbu (), au);

#if KLAYOUT_MAJOR_VERSION > 0 || (KLAYOUT_MAJOR_VERSION == 0 && KLAYOUT_MINOR_VERSION >= 27)
    std::pair<bool, unsigned int> lm = map.first_logical (db::LayerProperties (0, 0));
#else
    std::pair<bool, unsigned int> lm = map.logical (db::LayerProperties (0, 0));
#endif
    tl_assert (lm.first);

    EXPECT_EQ ((layout_au.cell (*layout_au.begin_top_down ()).bbox (lm.second) * au).to_string (), "(0,0;1000,1000)");

    std::string r = layout_meta_info_to_string (layout_au);
    EXPECT_EQ (r, "mebes_pattern_name[MEBES Pattern name]:MYPATXXXX.XX;mebes_maskshop_info[MEBES Maskshop info]:HELLO WORLD                             ;mebes_data_0[MEBES Data field 0]:mydata0;mebes_data_1[MEBES Data field 1]:;mebes_data_2[MEBES Data field 2]:mydata2x;mebes_date[MEBES file date (m/d/y)]:1/1/0;mebes_format[MEBES file format]:MODE5");
  }

  db::compare_layouts (_this, layout_au, testdata (au_file));
}

