
/*
  KLayout Layout Viewer
  Copyright (C) 2006-2019 Matthias Koefferlein
*/

#include "dbMEBESReader.h"
#include "dbTextWriter.h"
#include "tlUnitTest.h"
#include "tlFileUtils.h"

#include <stdlib.h>

static std::string testdata (const std::string &fn)
{
  return tl::testsrc () + "/src/plugins/mebes/testdata/" + fn;
}

void
compare_ref (tl::TestBase *_this, const char *test, const char *au_fn, const db::Layout &layout)
{
  tl::OutputStringStream os;
  tl::OutputStream ostream (os);
  db::TextWriter writer (ostream);
  writer.write (layout);
  std::string oss = os.string ();

  std::string fn_au (testdata (au_fn));

  std::string au;
  try {
    tl::InputFile is (fn_au);
    tl::InputStream istream (is);
    au = istream.read_all ();
  } catch (...) {
    //  ignore read errors on au files -> this way they can be updated easily
  }

  //  Normalize the golden data's CRLF line breaks on Windows
  au = tl::replaced (au, "\r\n", "\n");

  if (au != oss) {

    EXPECT_EQ (oss, au)

    std::string fn (_this->tmp_file (std::string ("t") + test + "_au.txt"));
    {
      tl::OutputFile ofs (fn);
      tl::OutputStream ofstream (ofs);
      ofstream.put (oss);
    }

    tl::info << "To update golden data use";
    tl::info << "  cp " << fn << " " << tl::absolute_file_path (fn_au);
#if 0 //  auto update
    system ((std::string ("cp ") + fn + " " + tl::absolute_file_path (fn_au)).c_str ());
    tl::info << "Golden data updated.";
#endif

  }
}

void
run_test (tl::TestBase *_this, const char *test, const char *in_fn, const db::LoadLayoutOptions &opt, db::Layout *playout = 0)
{
  db::Manager m;
  db::Layout layout (&m);
  if (! playout) {
    playout = &layout;
  }

  std::string fn (testdata (in_fn));
  tl::InputStream stream (fn);
  db::Reader reader (stream);
  reader.set_warnings_as_errors (true);

  bool error = false;
  try {
    reader.read (*playout, opt);
  } catch (tl::Exception &ex) {
    tl::error << ex.msg ();
    error = true;
  }
  EXPECT_EQ (error, false)

  compare_ref (_this, test, (std::string (test) + "_au.txt").c_str (), *playout);
}

std::string
layout_meta_info_to_string (const db::Layout &layout)
{
  std::string r;

  for (db::Layout::meta_info_iterator mi = layout.begin_meta (); mi != layout.end_meta (); ++mi) {
#if defined(KLAYOUT_META_INFO_V2)
    std::string name = layout.meta_info_name (mi->first);
    if (! r.empty ()) {
      r += ";";
    }
    r += name;
    r += "[";
    r += mi->second.description;
    r += "]:";
    r += mi->second.value.to_string ();
#else
    std::string name = mi->name;
    if (! r.empty ()) {
      r += ";";
    }
    r += name;
    r += "[";
    r += mi->description;
    r += "]:";
    r += mi->value;
#endif
  }

  return r;
}

TEST(mode5_1a)
{
  db::LoadLayoutOptions opt;
  run_test (_this, "mode5_1a", "mode5_1.meb", opt);
}

TEST(mode5_2a)
{
  db::LoadLayoutOptions opt;
  run_test (_this, "mode5_2a", "mode5_2.meb", opt);
}

TEST(mode5_1b)
{
  db::MEBESReaderOptions mebes_opt;
  mebes_opt.num_stripes_per_cell = 1;
  mebes_opt.produce_boundary = false;
  db::LoadLayoutOptions opt;
  opt.set_options (mebes_opt);

  run_test (_this, "mode5_1b", "mode5_1.meb", opt);
}

TEST(mode5_1c)
{
  db::MEBESReaderOptions mebes_opt;
  mebes_opt.subresolution = false;
  mebes_opt.num_stripes_per_cell = 1;
  mebes_opt.produce_boundary = false;
  db::LoadLayoutOptions opt;
  opt.set_options (mebes_opt);

  db::Manager m;
  db::Layout layout (&m);
  std::string fn = testdata ("mode5_1.meb");
  tl::InputStream stream (fn);
  db::Reader reader (stream);
  
  bool error = false;
  std::string emsg;
  try {
    db::LayerMap map = reader.read (layout, opt);
  } catch (tl::Exception &ex) {
    emsg = ex.msg ();
    error = true;
  }
  EXPECT_EQ (error, true)
  EXPECT_EQ (emsg, "Sub-resolution shape (dx1=168) (position=4114)")
}

TEST(mode5_2b)
{
  db::MEBESReaderOptions mebes_opt;
  mebes_opt.num_stripes_per_cell = 1;
  mebes_opt.produce_boundary = false;
  db::LoadLayoutOptions opt;
  opt.set_options (mebes_opt);

  run_test (_this, "mode5_2b", "mode5_2.meb", opt);
}

TEST(mode5_1d)
{
  db::MEBESReaderOptions mebes_opt;
  mebes_opt.num_stripes_per_cell = 1;
  mebes_opt.produce_boundary = false;
  mebes_opt.invert = true;
  db::LoadLayoutOptions opt;
  opt.set_options (mebes_opt);

  run_test (_this, "mode5_1d", "mode5_1.meb", opt);
}

TEST(mode5_2d)
{
  db::MEBESReaderOptions mebes_opt;
  mebes_opt.num_stripes_per_cell = 1;
  mebes_opt.produce_boundary = false;
  mebes_opt.invert = true;
  db::LoadLayoutOptions opt;
  opt.set_options (mebes_opt);

  run_test (_this, "mode5_2d", "mode5_2.meb", opt);
}

TEST(mode5_2e)
{
  db::MEBESReaderOptions mebes_opt;
  mebes_opt.num_stripes_per_cell = 1;
  mebes_opt.produce_boundary = true;
  mebes_opt.invert = true;
  db::LayerMap lm;
  lm.map (db::LayerProperties (0, 0), 0);
  mebes_opt.layer_map = lm;
  mebes_opt.create_other_layers = false;
  db::LoadLayoutOptions opt;
  opt.set_options (mebes_opt);

  db::Manager m;
  db::Layout layout (&m);
  run_test (_this, "mode5_2e", "mode5_2.meb", opt, &layout);

  std::string r = layout_meta_info_to_string (layout);
  EXPECT_EQ (r, "mebes_pattern_name[MEBES Pattern name]:XY0000000.00;mebes_maskshop_info[MEBES Maskshop info]:Some information    ;mebes_date[MEBES file date (m/d/y)]:6/14/2007;mebes_format[MEBES file format]:MODE5");
}

