
/*
  KLayout Layout Viewer
  Copyright (C) 2006-2019 Matthias Koefferlein
*/

#include "dbMEBESCompression.h"
#include "tlUnitTest.h"

#include <stdlib.h>
#include <set>

void expand (const db::DisplacementCompressor &data, std::set<db::Vector> &disp)
{
  for (db::DisplacementCompressor::disp_iterator d = data.begin_displacements (); d != data.end_displacements (); ++d) {
    disp.insert (*d);
  }

  for (db::DisplacementCompressor::array_iterator a = data.begin_arrays (); a != data.end_arrays (); ++a) {
    for (size_t i = 0; i < a->nx; ++i) {
      for (size_t j = 0; j < a->ny; ++j) {
        disp.insert (a->offset + db::Vector (i * a->dx, j * a->dy));
      }
    }
  }
}

bool compare_results (const db::DisplacementCompressor &a, const db::DisplacementCompressor &b)
{
  std::set<db::Vector> a_disp, b_disp;
  expand (a, a_disp);
  expand (b, b_disp);

  bool equal = true;
  for (std::set<db::Vector>::const_iterator i = a_disp.begin (); i != a_disp.end (); ++i) {
    if (b_disp.find (*i) == b_disp.end ()) {
      tl::info << "Not in b, but in a: " << i->to_string ();
      equal = false;
    }
  }
  for (std::set<db::Vector>::const_iterator i = b_disp.begin (); i != b_disp.end (); ++i) {
    if (a_disp.find (*i) == a_disp.end ()) {
      tl::info << "Not in a, but in b: " << i->to_string ();
      equal = false;
    }
  }

  if (! equal) {

    tl::info << "List of A:";
    for (std::set<db::Vector>::const_iterator i = a_disp.begin (); i != a_disp.end (); ++i) {
      tl::info << i->to_string () << " " << tl::noendl;
    }
    tl::info << "";

    tl::info << "List of B:";
    for (std::set<db::Vector>::const_iterator i = b_disp.begin (); i != b_disp.end (); ++i) {
      tl::info << i->to_string () << " " << tl::noendl;
    }
    tl::info << "";

  }

  return equal;
}

double compression_statistics (int level, const db::DisplacementCompressor &data)
{
  size_t total = 0;
  size_t arrays = 0, single = 0;
  size_t objects = 0;
  size_t weighted_objects = 0;
  size_t single_arrays = 0;

  for (db::DisplacementCompressor::disp_iterator d = data.begin_displacements (); d != data.end_displacements (); ++d) {
    total += 1;
    single += 1;
    objects += 1;
    weighted_objects += 1;
  }

  for (db::DisplacementCompressor::array_iterator a = data.begin_arrays (); a != data.end_arrays (); ++a) {
    total += a->nx * a->ny;
    if (a->nx * a->ny == 1) {
      single_arrays += 1;
    }
    arrays += 1;
    objects += 1;
    weighted_objects += 2;
  }

  bool print_stat = false;
  if (print_stat) {
    tl::info << "Level " << level
              << ": total = " << total
              << " (" << single << "s/" << arrays << "a/" << single_arrays << "sa)"
              << ", objects = " << objects
              << ", weighted = " << weighted_objects
              << ", ratio = " << double (weighted_objects) / double (total);
  }

  return double (weighted_objects) / double (total);
}

TEST(1)
{
  double l0stat = 0, l1stat = 0, l2stat = 0, l10stat = 0;
  size_t num = 0;

  for (unsigned int seed = 0; seed < 100; ++seed) {

    db::DisplacementCompressor compressor;

    srand (seed);

    for (int i = 0; i < 100; ++i) {
      int x = rand () % 50;
      int y = rand () % 50;
      compressor.add (db::Vector (x, y));
    }

    db::DisplacementCompressor org = compressor;

    compressor.compress (0, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l0stat += compression_statistics (0, compressor);

    compressor = org;
    compressor.compress (1, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l1stat += compression_statistics (1, compressor);

    compressor = org;
    compressor.compress (2, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l2stat += compression_statistics (2, compressor);

    compressor = org;
    compressor.compress (10, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l10stat += compression_statistics (10, compressor);

    ++num;

  }

  tl::info << "Average compression ratio (weighted):";
  tl::info << "  L0:  " << l0stat / num;
  tl::info << "  L1:  " << l1stat / num;
  tl::info << "  L2:  " << l2stat / num;
  tl::info << "  L10: " << l10stat / num;
}

TEST(2)
{
  double l0stat = 0, l1stat = 0, l2stat = 0, l10stat = 0;
  size_t num = 0;

  for (unsigned int seed = 0; seed < 100; ++seed) {

    db::DisplacementCompressor compressor;

    srand (seed);

    for (int i = 0; i < 10; ++i) {
      int x0 = 0, y0 = 0;
      if (rand () % 2 == 0) {
        x0 = rand () % 50;
      } else {
        y0 = rand () % 50;
      }
      for (int j = 0; j < 10; ++j) {
        for (int k = 0; k < 10; ++k) {
          compressor.add (db::Vector (x0 + j * 50, y0 + k * 50));
        }
      }
    }

    db::DisplacementCompressor org = compressor;

    compressor.compress (0, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l0stat += compression_statistics (0, compressor);

    compressor = org;
    compressor.compress (1, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l1stat += compression_statistics (1, compressor);

    compressor = org;
    compressor.compress (2, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l2stat += compression_statistics (2, compressor);

    compressor = org;
    compressor.compress (10, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l10stat += compression_statistics (10, compressor);

    ++num;

  }

  tl::info << "Average compression ratio (weighted):";
  tl::info << "  L0:  " << l0stat / num;
  tl::info << "  L1:  " << l1stat / num;
  tl::info << "  L2:  " << l2stat / num;
  tl::info << "  L10: " << l10stat / num;
}

TEST(3)
{
  double l0stat = 0, l1stat = 0, l2stat = 0, l10stat = 0;
  double l100stat = 0, l101stat = 0, l102stat = 0, l110stat = 0;
  size_t num = 0;

  for (unsigned int seed = 0; seed < 100; ++seed) {

    db::DisplacementCompressor compressor;

    srand (seed);

    int x0 = 0, y0 = 0;
    for (int j = 0; j < 20; ++j) {
      for (int k = 0; k < 20; ++k) {
        if ((rand () % 10) != 0) {
          compressor.add (db::Vector (x0 + j * 50, y0 + k * 50));
        }
      }
    }

    db::DisplacementCompressor org = compressor;

    compressor.compress (0, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l0stat += compression_statistics (0, compressor);

    compressor = org;
    compressor.compress (1, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l1stat += compression_statistics (1, compressor);

    compressor = org;
    compressor.compress (2, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l2stat += compression_statistics (2, compressor);

    compressor = org;
    compressor.compress (10, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l10stat += compression_statistics (10, compressor);

    compressor = org;
    compressor.compress (100, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l100stat += compression_statistics (100, compressor);

    compressor = org;
    compressor.compress (101, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l101stat += compression_statistics (101, compressor);

    compressor = org;
    compressor.compress (102, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l102stat += compression_statistics (102, compressor);

    compressor = org;
    compressor.compress (110, false);
    EXPECT_EQ (compare_results (org, compressor), true);

    l110stat += compression_statistics (110, compressor);

    ++num;

  }

  tl::info << "Average compression ratio (weighted):";
  tl::info << "  L0:   " << l0stat / num;
  tl::info << "  L1:   " << l1stat / num;
  tl::info << "  L2:   " << l2stat / num;
  tl::info << "  L10:  " << l10stat / num;
  tl::info << "  L100: " << l100stat / num;
  tl::info << "  L101: " << l101stat / num;
  tl::info << "  L102: " << l102stat / num;
  tl::info << "  L110: " << l110stat / num;
}
