
/*
  KLayout Layout Viewer
  Copyright (C) 2006-2019 Matthias Koefferlein
*/

#include "gsiDecl.h"
#include "dbCommon.h"
#include "dbMEBESReader.h"
#include "dbMEBESWriter.h"

namespace gsi
{

// ---------------------------------------------------------------
//  Reader GSI declarations

static void set_mebes_invert (db::LoadLayoutOptions *options, bool inv)
{
  options->get_options<db::MEBESReaderOptions> ().invert = inv;
}

static bool get_mebes_invert (const db::LoadLayoutOptions *options)
{
  return options->get_options<db::MEBESReaderOptions> ().invert;
}

static void set_mebes_num_stripes_per_cell (db::LoadLayoutOptions *options, int n)
{
  options->get_options<db::MEBESReaderOptions> ().num_stripes_per_cell = n;
}

static int get_mebes_num_stripes_per_cell (const db::LoadLayoutOptions *options)
{
  return options->get_options<db::MEBESReaderOptions> ().num_stripes_per_cell;
}

static void set_mebes_num_shapes_per_cell (db::LoadLayoutOptions *options, int n)
{
  options->get_options<db::MEBESReaderOptions> ().num_shapes_per_cell = n;
}

static int get_mebes_num_shapes_per_cell (const db::LoadLayoutOptions *options)
{
  return options->get_options<db::MEBESReaderOptions> ().num_shapes_per_cell;
}

static void set_mebes_subresolution (db::LoadLayoutOptions *options, bool f)
{
  options->get_options<db::MEBESReaderOptions> ().subresolution = f;
}

static bool get_mebes_subresolution (const db::LoadLayoutOptions *options)
{
  return options->get_options<db::MEBESReaderOptions> ().subresolution;
}

static void set_mebes_produce_boundary (db::LoadLayoutOptions *options, bool f)
{
  options->get_options<db::MEBESReaderOptions> ().produce_boundary = f;
}

static bool get_mebes_produce_boundary (const db::LoadLayoutOptions *options)
{
  return options->get_options<db::MEBESReaderOptions> ().produce_boundary;
}

static void set_mebes_data_layer (db::LoadLayoutOptions *options, int layer)
{
  options->get_options<db::MEBESReaderOptions> ().data_layer = layer;
}

static int get_mebes_data_layer (const db::LoadLayoutOptions *options)
{
  return options->get_options<db::MEBESReaderOptions> ().data_layer;
}

static void set_mebes_data_datatype (db::LoadLayoutOptions *options, int datatype)
{
  options->get_options<db::MEBESReaderOptions> ().data_datatype = datatype;
}

static int get_mebes_data_datatype (const db::LoadLayoutOptions *options)
{
  return options->get_options<db::MEBESReaderOptions> ().data_datatype;
}

static void set_mebes_data_name (db::LoadLayoutOptions *options, const std::string &name)
{
  options->get_options<db::MEBESReaderOptions> ().data_name = name;
}

static std::string get_mebes_data_name (const db::LoadLayoutOptions *options)
{
  return options->get_options<db::MEBESReaderOptions> ().data_name;
}

static void set_mebes_boundary_layer (db::LoadLayoutOptions *options, int layer)
{
  options->get_options<db::MEBESReaderOptions> ().boundary_layer = layer;
}

static int get_mebes_boundary_layer (const db::LoadLayoutOptions *options)
{
  return options->get_options<db::MEBESReaderOptions> ().boundary_layer;
}

static void set_mebes_boundary_datatype (db::LoadLayoutOptions *options, int datatype)
{
  options->get_options<db::MEBESReaderOptions> ().boundary_datatype = datatype;
}

static int get_mebes_boundary_datatype (const db::LoadLayoutOptions *options)
{
  return options->get_options<db::MEBESReaderOptions> ().boundary_datatype;
}

static void set_mebes_boundary_name (db::LoadLayoutOptions *options, const std::string &name)
{
  options->get_options<db::MEBESReaderOptions> ().boundary_name = name;
}

static std::string get_mebes_boundary_name (const db::LoadLayoutOptions *options)
{
  return options->get_options<db::MEBESReaderOptions> ().boundary_name;
}

static void set_mebes_top_cell_index (db::LoadLayoutOptions *options, int cell_index)
{
  options->get_options<db::MEBESReaderOptions> ().top_cell_index = cell_index;
}

static int get_mebes_top_cell_index (const db::LoadLayoutOptions *options)
{
  return options->get_options<db::MEBESReaderOptions> ().top_cell_index;
}

static void set_layer_map (db::LoadLayoutOptions *options, const db::LayerMap &lm, bool f)
{
  options->get_options<db::MEBESReaderOptions> ().layer_map = lm;
  options->get_options<db::MEBESReaderOptions> ().create_other_layers = f;
}

static void set_layer_map1 (db::LoadLayoutOptions *options, const db::LayerMap &lm)
{
  options->get_options<db::MEBESReaderOptions> ().layer_map = lm;
}

static db::LayerMap get_layer_map (const db::LoadLayoutOptions *options)
{
  return options->get_options<db::MEBESReaderOptions> ().layer_map;
}

static void select_all_layers (db::LoadLayoutOptions *options)
{
  options->get_options<db::MEBESReaderOptions> ().layer_map = db::LayerMap ();
  options->get_options<db::MEBESReaderOptions> ().create_other_layers = true;
}

static bool create_other_layers (const db::LoadLayoutOptions *options)
{
  return options->get_options<db::MEBESReaderOptions> ().create_other_layers;
}

static void set_create_other_layers (db::LoadLayoutOptions *options, bool l)
{
  options->get_options<db::MEBESReaderOptions> ().create_other_layers = l;
}

//  extend lay::LoadLayoutOptions with the GDS2 options
static
gsi::ClassExt<db::LoadLayoutOptions> mebes_writer_options (
  gsi::method_ext ("mebes_set_layer_map", &set_layer_map, gsi::arg ("map"), gsi::arg ("create_other_layers"),
    "@brief Sets the layer map\n"
      "This sets a layer mapping for the reader. The layer map allows selection and translation of the original layers.\n"
    "@param map The layer map to set.\n"
    "@param create_other_layers The flag indicating whether other layers will be created as well. Set to false to read only the layers in the layer map.\n"
    "\n"
    "This method has been added in version 0.25 and replaces the respective global option in \\LoadLayoutOptions "
    "in a format-specific fashion."
  ) +
  gsi::method_ext ("mebes_select_all_layers", &select_all_layers,
    "@brief Selects all layers and disables the layer map\n"
    "\n"
    "This disables any layer map and enables reading of all layers.\n"
    "New layers will be created when required.\n"
    "\n"
    "This method has been added in version 0.25 and replaces the respective global option in \\LoadLayoutOptions "
    "in a format-specific fashion."
  ) +
  gsi::method_ext ("mebes_layer_map", &get_layer_map,
    "@brief Gets the layer map\n"
    "@return The layer map.\n"
    "\n"
    "This method has been added in version 0.25 and replaces the respective global option in \\LoadLayoutOptions "
    "in a format-specific fashion."
  ) +
  gsi::method_ext ("mebes_layer_map=", &set_layer_map1, gsi::arg ("map"),
    "@brief Sets the layer map\n"
    "This sets a layer mapping for the reader. Unlike \\mebes_set_layer_map, the 'create_other_layers' flag is not changed.\n"
    "@param map The layer map to set.\n"
    "\n"
    "This convenience method has been added in version 0.26.2."
  ) +
  gsi::method_ext ("mebes_create_other_layers?", &create_other_layers,
    "@brief Gets a value indicating whether other layers shall be created\n"
    "@return True, if other layers will be created.\n"
    "This attribute acts together with a layer map (see \\mebes_layer_map=). Layers not listed in this map are created as well when "
    "\\mebes_create_other_layers? is true. Otherwise they are ignored.\n"
    "\n"
    "This method has been added in version 0.25 and replaces the respective global option in \\LoadLayoutOptions "
    "in a format-specific fashion."
  ) +
  gsi::method_ext ("mebes_create_other_layers=", &set_create_other_layers, gsi::arg ("create"),
    "@brief Specifies whether other layers shall be created\n"
    "@param create True, if other layers will be created.\n"
    "See \\mebes_create_other_layers? for a description of this attribute.\n"
    "\n"
    "This method has been added in version 0.25 and replaces the respective global option in \\LoadLayoutOptions "
    "in a format-specific fashion."
  ) +
  gsi::method_ext ("mebes_top_cell_index=", &set_mebes_top_cell_index, gsi::arg ("cell_index"),
    "@brief Specify the cell index for the top cell to use\n"
    "If this property is set to a valid cell index, the MEBES reader will put the "
    "subcells and shapes into this cell.\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_top_cell_index", &get_mebes_top_cell_index,
    "@brief Gets the cell index for the top cell to use\n"
    "See \\mebes_top_cell_index= for a description of this property.\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_invert=", &set_mebes_invert, gsi::arg ("flag"),
    "@brief Specify whether to invert the MEBES pattern\n"
    "If this property is set to true, the pattern will be inverted.\n"
    "\nThis property has been added in version 0.22.\n"
  ) +
  gsi::method_ext ("mebes_invert", &get_mebes_invert,
    "@brief Gets a value indicating whether to invert the MEBES pattern\n"
    "If this property is set to true, the pattern will be inverted.\n"
    "\nThis property has been added in version 0.22.\n"
  ) +
  gsi::method_ext ("mebes_data_layer=", &set_mebes_data_layer, gsi::arg ("layer"),
    "@brief Sets the layer number of the data layer to produce\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_data_layer", &get_mebes_data_layer,
    "@brief Gets the layer number of the data layer to produce\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_data_datatype=", &set_mebes_data_datatype, gsi::arg ("datatype"),
    "@brief Sets the datatype number of the data layer to produce\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_data_datatype", &get_mebes_data_datatype,
    "@brief Gets the datatype number of the data layer to produce\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_data_name=", &set_mebes_data_name, gsi::arg ("name"),
    "@brief Sets the name of the data layer to produce\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_data_name", &get_mebes_data_name,
    "@brief Gets the name of the data layer to produce\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_boundary_layer=", &set_mebes_boundary_layer, gsi::arg ("layer"),
    "@brief Sets the layer number of the boundary layer to produce\n"
    "See \\mebes_produce_boundary= for a description of this attribute.\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_boundary_layer", &get_mebes_boundary_layer,
    "@brief Gets the layer number of the boundary layer to produce\n"
    "See \\mebes_produce_boundary= for a description of this attribute.\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_boundary_datatype=", &set_mebes_boundary_datatype, gsi::arg ("datatype"),
    "@brief Sets the datatype number of the boundary layer to produce\n"
    "See \\mebes_produce_boundary= for a description of this attribute.\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_boundary_datatype", &get_mebes_boundary_datatype,
    "@brief Gets the datatype number of the boundary layer to produce\n"
    "See \\mebes_produce_boundary= for a description of this attribute.\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_boundary_name=", &set_mebes_boundary_name, gsi::arg ("name"),
    "@brief Sets the name of the boundary layer to produce\n"
    "See \\mebes_produce_boundary= for a description of this attribute.\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_boundary_name", &get_mebes_boundary_name,
    "@brief Gets the name of the boundary layer to produce\n"
    "See \\mebes_produce_boundary= for a description of this attribute.\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_produce_boundary=", &set_mebes_produce_boundary, gsi::arg ("flag"),
    "@brief Specify whether to produce a boundary layer\n"
    "If this property is set to true, the pattern boundary will be written to the layer and datatype specified with \\mebes_boundary_name, \\mebes_boundary_layer and \\mebes_boundary_datatype.\n"
    "By default, the boundary layer is produced.\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_produce_boundary", &get_mebes_produce_boundary,
    "@brief Gets a value indicating whether a boundary layer will be produced\n"
    "See \\mebes_produce_boundary= for details about this property.\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_num_stripes_per_cell=", &set_mebes_num_stripes_per_cell, gsi::arg ("n"),
    "@brief Specify the number of stripes collected per cell\n"
    "This property specifies how many stripes will be collected into one cell.\n"
    "A smaller value means less but bigger cells. The default value is 64.\n"
    "New cells will be formed whenever more than this number of stripes has been read\n"
    "or a new segment is started and the number of shapes given by \\mebes_num_shapes_per_cell\n"
    "is exceeded.\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_num_stripes_per_cell", &get_mebes_num_stripes_per_cell,
    "@brief Gets the number of stripes collected per cell\n"
    "See \\mebes_num_stripes_per_cell= for details about this property.\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_num_shapes_per_cell=", &set_mebes_num_shapes_per_cell, gsi::arg ("n"),
    "@brief Specify the number of stripes collected per cell\n"
    "See \\mebes_num_stripes_per_cell= for details about this property.\n"
    "\nThis property has been added in version 0.24.5.\n"
  ) +
  gsi::method_ext ("mebes_num_shapes_per_cell", &get_mebes_num_shapes_per_cell,
    "@brief Gets the number of stripes collected per cell\n"
    "See \\mebes_num_stripes_per_cell= for details about this property.\n"
    "\nThis property has been added in version 0.24.5.\n"
  ) +
  gsi::method_ext ("mebes_subresolution=", &set_mebes_subresolution, gsi::arg ("flag"),
    "@brief Specify whether subresolution trapezoids are supported\n"
    "If this property is set to true, subresolution trapezoid vertices are supported.\n"
    "In order to implement support, the reader will create magnified instances with a magnification of 1/16.\n"
    "By default this property is enabled.\n"
    "\nThis property has been added in version 0.23.10.\n"
  ) +
  gsi::method_ext ("mebes_subresolution", &get_mebes_subresolution,
    "@brief Gets a value indicating whether to invert the MEBES pattern\n"
    "See \\subresolution= for details about this property.\n"
    "\nThis property has been added in version 0.23.10.\n"
  ),
  ""
);

// ---------------------------------------------------------------
//  Writer GSI declarations

//  extend lay::LoadLayoutOptions with the GDS2 options
static
gsi::Class<db::FracturedData> decl_fractured_data ("mebes", "MEBESFracturedData",
  gsi::Methods (),
  "@brief An opaque class that represents one tile of fractured data\n"
  "This object is produced by \\MEBESWriter#fracture and can be passed\n"
  "to the MEBES writer from inside the tiling processor script. This enables\n"
  "implementation of multithreaded fracture schemes.\n"
  "\n"
  "This object can been introduced in version 0.25."
);

static db::MEBESWriter *create_mebes_writer (const std::string &path)
{
  return new db::MEBESWriter (path);
}

DB_PUBLIC gsi::Class<db::TileOutputReceiver> &dbdecl_TileOutputReceiverBase ();

static
gsi::Class<db::MEBESWriter> mebes_writer (dbdecl_TileOutputReceiverBase (), "mebes", "MEBESWriter",
  gsi::constructor ("new", &create_mebes_writer, gsi::arg ("path"),
    "@brief Creates a MEBES fracture engine instance\n"
    "@param path The path to the output file to be written.\n"
  ) +
  gsi::method ("fracture", &db::MEBESWriter::fracture, gsi::arg ("region"), gsi::arg ("tile"), gsi::arg ("dbu"),
    "@brief Produces fractured data for the tile\n"
    "The fractured data can be output from the tiling processor's work script. It is then sent to the writer and "
    "finally written to the MEBES file.\n"
  ) +
  gsi::method ("subresolution=", &db::MEBESWriter::enable_subresolution_fracturing, gsi::arg ("subres"),
    "@brief Sets a value enabling subresolution fracturing\n"
  ) +
  gsi::method ("subresolution", &db::MEBESWriter::is_subresolution_fracturing_enabled,
    "@brief Gets a value indicating subresolution fracturing\n"
  ) +
  gsi::method ("smoothing=", &db::MEBESWriter::enable_smoothing, gsi::arg ("smoothing_enabled"),
    "@brief Sets a value enabling integrated polygon smoothing\n"
  ) +
  gsi::method ("smoothing", &db::MEBESWriter::is_smoothing_enabled,
    "@brief Gets a value indicating integrated smoothing\n"
  ) +
  gsi::method ("fracture_caching=", &db::MEBESWriter::enable_fracture_caching, gsi::arg ("caching enabled"),
    "@brief Sets a value enabling fracture caching\n"
  ) +
  gsi::method ("fracture_caching", &db::MEBESWriter::is_fracture_caching_enabled,
    "@brief Gets a value indicating fracture caching\n"
  ) +
  gsi::method ("pattern_name=", &db::MEBESWriter::set_pattern_name, gsi::arg ("name"),
    "@brief Sets the pattern name data field of the output MEBES file\n"
  ) +
  gsi::method ("pattern_name", &db::MEBESWriter::pattern_name,
    "@brief Gets the pattern name data field of the output MEBES file\n"
  ) +
  gsi::method ("set_data", &db::MEBESWriter::set_data, gsi::arg ("n"), gsi::arg ("text"),
    "@brief Sets the nth data field of the output MEBES file\n"
  ) +
  gsi::method ("data", &db::MEBESWriter::data, gsi::arg ("n"),
    "@brief Gets the nth data field of the output MEBES file\n"
  ) +
  gsi::method ("maskshop_info=", &db::MEBESWriter::set_maskshop_info, gsi::arg ("info"),
    "@brief Sets the maskshop info data field of the output MEBES file\n"
  ) +
  gsi::method ("maskshop_info", &db::MEBESWriter::maskshop_info,
    "@brief Gets the maskshop info data field of the output MEBES file\n"
  ) +
  gsi::method ("address_unit=", &db::MEBESWriter::set_address_unit, gsi::arg ("au"),
    "@brief Sets the address unit of the output MEBES file (in micron units)\n"
  ) +
  gsi::method ("address_unit", &db::MEBESWriter::get_address_unit,
    "@brief Gets the address unit of the output MEBES file (in micron units)\n"
  ) +
  gsi::method ("compression_level=", &db::MEBESWriter::set_compression_level, gsi::arg ("level"),
    "@brief Sets the compression level of the writer\n"
    "A level value of 0 means no compression. A level of 1 means weak compression. "
    "Higher levels mean more effort for compression. A level of 2 is the recommended and default value."
  ) +
  gsi::method ("compression_level", &db::MEBESWriter::get_compression_level,
    "@brief Gets the compression level of the writer\n"
  ),
  "@brief A MEBES writer implementation\n"
  "This MEBES writer is both a file production class and a tile processor receiver.\n"
  "Here is some sample code how to use the MEBES writer:\n"
  "\n"
  "@code\n"
  "au = 0.01    # fracture address unit\n"
  "\n"
  "writer = RBA::MEBESWriter::new(\"OUT.MEB\")\n"
  "writer.address_unit = au\n"
  "\n"
  "proc = RBA::TilingProcessor::new\n"
  "proc.input(\"input\", layout.top_cell, layer)\n"
  "proc.output(\"writer\", writer)\n"
  "proc.dbu = au\n"
  "proc.tile_size(32768*au, 65536*au)\n"
  "\n"
  "proc.queue(\"_output(writer, _rec(writer).fracture(input, _tile.bbox, _dbu)\")\n"
  "\n"
  "proc.execute\n"
  "@/code\n"
  "\n"
  "This class has been introduced in version 0.25."
);

}
