
/*
  KLayout Layout Viewer
  Copyright (C) 2006-2019 Matthias Koefferlein
*/

#ifndef HDR_dbMEBESWriter
#define HDR_dbMEBESWriter

#include "dbTilingProcessor.h"
#include "dbMEBESCompression.h"
#include "dbHash.h"
#include "tlStream.h"
#include "tlTimer.h"
#include "gsiObject.h"

namespace db
{

struct TrapezoidArraySpec;

/**
 *  @brief A class representing fractured data
 *
 *  This class is opaque and is delivered by the MEBES writer's fracture method.
 *  It can be wrapped into a tl::Variant so it can be passed through the tiling
 *  processor's interfaces.
 */
class FracturedData
  : public gsi::ObjectBase
{
public:
  FracturedData () { }

  void swap (FracturedData &other)
  {
    normalized.swap (other.normalized);
    arrays.swap (other.arrays);
  }

  typedef db::DisplacementCompressor::disp_vector disp_vector;
  typedef db::DisplacementCompressor::array_vector array_vector;

  std::unordered_map<db::SimplePolygon, disp_vector> normalized;
  std::vector<std::pair<db::SimplePolygon, array_vector> > arrays;
};

/**
 *  @brief The MEBES writer class
 *
 *  The MEBES writer in fact is a tiling processor receiver. The processor feeding the writer
 *  has to be configured to deliver tiles which are compatible with the MEBES format's capabilities.
 *  Also the writer relies on the fact, that tiles are delivers bottom-up and left to right.
 *
 *  The MEBES writer can only accept one region object per tile.
 */
class DB_PLUGIN_PUBLIC MEBESWriter
  : public db::TileOutputReceiver
{
public:
  /**
   *  @brief Constructor
   *  @param path The path to the file to write
   */
  MEBESWriter (const std::string &path);

  /**
   *  @brief Destructor
   */
  ~MEBESWriter ();

  /**
   *  @brief Enables fracture caching
   *
   *  If this option is enabled, the fracturing results of polygons
   *  is cached. Caching will consume some memory.
   *  The default setting is "off".
   */
  void enable_fracture_caching (bool en);

  /**
   *  @brief Gets a value indicating whether fracture caching is enabled
   */
  bool is_fracture_caching_enabled () const
  {
    return m_fracture_caching_enabled;
  }
  
  /**
   *  @brief Enables the header's time stamp
   * 
   *  If this option is set, the file header will contain the current date.
   *  Otherwise a date of "1.1.0" is entered. Disabling the time stamp may
   *  be desirable if binary identity of files produced at different times
   *  is required.
   *  The default setting is "on".
   */
  void enable_timestamp (bool en);
  
  /**
   *  @brief Gets a value indicating whether the timestamp is enabled
   */
  bool is_timestamp_enabled () const
  {
    return m_timestamp_enabled;
  }

  /**
   *  @brief Enables polygon smoothing
   *
   *  If this option is enabled, polygon will be smoothed prior to fracturing.
   *  Smoothing leads to somewhat better fracturing result since excess
   *  vertices are removed.
   *  The default setting in "on".
   */
  void enable_smoothing (bool en);

  /**
   *  @brief Gets a value indicating whether polygon smooting is enabled
   */
  bool is_smoothing_enabled () const
  {
    return m_smoothing_enabled;
  }

  /**
   *  @brief Enables subresolution fracturing
   *
   *  If subresolution fracturing is enabled, trapezoids with a resolution
   *  of 1/16th of the address unit are produced.
   *  The default setting is "off".
   */
  void enable_subresolution_fracturing (bool en);

  /**
   *  @brief Gets a value indicating whether subresolution fracturing is enabled
   */
  bool is_subresolution_fracturing_enabled () const
  {
    return m_subresolution_fracturing_enabled;
  }

  /**
   *  @brief Sets the pattern name being issued as the pattern name header field
   *
   *  This name is issues as the pattern name header field. The value needs to be
   *  a name conforming to the 9.2 format suggested by MEBES.
   */
  void set_pattern_name (const std::string &s);

  /**
   *  @brief Gets the patter name
   */
  const std::string &pattern_name () const
  {
    return m_pattern_name;
  }

  /**
   *  @brief Sets the string for the nth additional data field
   *
   *  @param n The index of the data field
   *  @param s The string to write
   *
   *  n is the index of the data field - zero means the first one which comes after
   *  the mask shop info.
   */
  void set_data (int n, const std::string &s);

  /**
   *  @brief Gets the string for the nth additional data field
   *
   *  @param n The index of the data field
   *  @return The value
   *
   *  See "set_data" for details about the additional data field
   */
  const std::string &data (int n) const;

  /**
   *  @brief Sets the maskshop info
   *
   *  The maskshop info is a 40 character max string that is printed into the
   *  maskshop info header field.
   */
  void set_maskshop_info (const std::string &s);

  /**
   *  @brief Gets the maskshop info
   */
  const std::string &maskshop_info () const
  {
    return m_maskshop_info;
  }

  /**
   *  @brief Specifies the address unit in micron units
   *  @param unit The address unit
   *
   *  The address unit is used to compute the segment and stripe size.
   *  The tiling processor has to be configured with dimensions compatible with the
   *  capabilities of the MEBES format.
   */
  void set_address_unit (double unit);

  /**
   *  @brief Gets the address unit
   */
  double get_address_unit () const
  {
    return m_address_unit;
  }

  /**
   *  @brief Specifies the compression level
   *
   *  0 is "no compression", 1 and further means taking nth next neightbors for
   *  array formation. Higher values take longer for compression.
   */
  void set_compression_level (unsigned int level);

  /**
   *  @brief Gets the compression level
   */
  unsigned int get_compression_level () const
  {
    return m_compression_level;
  }

  /**
   *  @brief Fractures the given region
   *
   *  This method allows one to do fracturing inside the tiling processor (hence multithreaded).
   *  It delivers a FracturedData object which can be output by the tiling processor
   *  through the tl::Variant object.
   */
  FracturedData fracture (const db::Region &region, const db::Box &tile, double dbu) const;

  /**
   *  @brief Implementation of the TileOutputReceiver interface: starts execution
   */
  virtual void begin (size_t nx, size_t ny, const db::DPoint &p0, double dx, double dy, const db::DBox &frame);

  /**
   *  @brief Implementation of the TileOutputReceiver interface: deliver an object for one tile
   */
  virtual void put (size_t ix, size_t iy, const db::Box &tile, size_t id, const tl::Variant &obj, double dbu, const db::ICplxTrans &trans, bool clip);

  /**
   *  @brief Implementation of the TileOutputReceiver interface: indicates the end of the run
   */
  virtual void finish (bool success);

private:
  std::map<std::pair<size_t, size_t>, FracturedData> m_tile_storage;
  size_t m_next_segment, m_next_stripe;
  size_t m_segments, m_stripes, m_cluster_size;
  bool m_timestamp_enabled;
  bool m_fracture_caching_enabled;
  bool m_subresolution_fracturing_enabled;
  bool m_smoothing_enabled;
  std::string m_pattern_name, m_maskshop_info;
  std::map<int, std::string> m_data;
  db::DPoint m_offset;
  std::unique_ptr<tl::OutputStream> mp_stream;
  double m_address_unit;
  unsigned int m_compression_level;
  size_t m_max_storage_size;
  tl::SelfTimer m_timer;
  std::string m_path;
  db::DBox m_frame;
  size_t m_directory_pos;
  std::vector<unsigned int> m_directory;
  bool m_has_errors;
  bool m_start_written;
  bool m_needs_new_segment;

  void write_figure (const db::SimplePolygon &poly, const TrapezoidArraySpec &ta);
  void write_array (const TrapezoidArraySpec &ta);
  void deliver_tile (const FracturedData &data);
  void deliver_stored ();
  void finish_record ();
  void require_bytes (size_t n);
  void write_word (unsigned int w);
  void write_dword (unsigned int w);
  void before_first_segment ();
  void before_stripe ();
  void before_figure ();
  void after_stripe ();
};

}

#if KLAYOUT_MAJOR_VERSION == 0 && KLAYOUT_MINOR_VERSION < 28

namespace tl
{
  template <>
  struct type_traits<db::MEBESWriter>
    : public type_traits<void>
  {
    typedef tl::false_tag has_copy_constructor;
    typedef tl::false_tag has_default_constructor;
  };
}

#endif

#endif
