
/*
  KLayout Layout Viewer
  Copyright (C) 2006-2019 Matthias Koefferlein
*/

#ifndef HDR_dbMEBESReader
#define HDR_dbMEBESReader

#include "dbPluginCommon.h"

#include "tlException.h"
#include "tlInternational.h"
#include "tlProgress.h"
#include "tlString.h"

#include "dbLayout.h"
#include "dbShapeProcessor.h"
#include "dbReader.h"
#include "tlStream.h"

namespace db
{

/**
 *  @brief Structure that holds the MEBES specific options for the reader
 */
class DB_PLUGIN_PUBLIC MEBESReaderOptions
  : public FormatSpecificReaderOptions
{
public:
  /**
   *  @brief The constructor
   */
  MEBESReaderOptions ()
    : invert (false),
      produce_boundary (true),
      subresolution (true),
      num_stripes_per_cell (64),
      num_shapes_per_cell (0),
      data_layer (1),
      data_datatype (0),
      data_name ("DATA"),
      boundary_layer (0),
      boundary_datatype (0),
      boundary_name ("BORDER"),
      top_cell_index (-1),
      create_other_layers (true)
  {
    //  .. nothing yet ..
  }

  /**
   *  @brief Invert the layout
   *
   *  If this property is true, the layout will be inverted.
   */
  bool invert; 

  /**
   *  @brief Produce a boundary layer
   *
   *  If this property is true, a boundary rectangle will be produced on 
   *  the layer given by boundary_layer, boundary_datatype and boundary_name.
   */
  bool produce_boundary; 

  /**
   *  @brief Produce subresolution trapezoids
   *
   *  If this property is true, subresolution trapezoids are supported.
   *  For this support, it is required to produce magnified instances
   *  with a magnification of 1/16.
   */
  bool subresolution;

  /**
   *  @brief The number of stripes per cell
   *
   *  This value specifies the number of stripes generated per cell.
   *  Higher values lead to a lower number of cells but bigger cells.
   *  The default value is 64. If the value is 0, all stripes of a 
   *  segment are put into the same cell. If the num_shapes_per_cell
   *  value is non-null, a new cell will be started when the specified
   *  number of shapes is present in a cell.
   */
  int num_stripes_per_cell;

  /**
   *  @brief The number of shapes per cell
   *
   *  This value specifies the minimum number of shapes per cell. 
   *  If the value is 0, no such minimum is present and the 
   *  num_stripes_per_cell value controls the formation of cells.
   *  The default value is 0.
   */
  int num_shapes_per_cell;

  /**
   *  @brief The layer number of the data layer
   */
  int data_layer;

  /**
   *  @brief The datatype number of the data layer
   */
  int data_datatype;

  /**
   *  @brief The name of the data layer
   */
  std::string data_name;

  /**
   *  @brief The layer number of the boundary layer
   */
  int boundary_layer;

  /**
   *  @brief The datatype number of the boundary layer
   */
  int boundary_datatype;

  /**
   *  @brief The name of the boundary layer
   */
  std::string boundary_name;

  /**
   *  @brief The index of the cell to use as the top cell
   *
   *  This option allows speficiation of the top cell to use.
   *  By default, a cell named "MEBES_TOP" is created. By using this
   *  option, an existing cell can be used.
   */
  int top_cell_index;

  /**
   *  @brief Specifies a layer mapping
   *
   *  If a layer mapping is specified, only the given layers are read.
   *  Otherwise, all layers are read.
   *  Setting "create_other_layers" to true will make the reader
   *  create other layers for all layers not given in the layer map.
   *  Setting an empty layer map and create_other_layers to true effectively
   *  enables all layers for reading.
   */
  db::LayerMap layer_map;

  /**
   *  @brief A flag indicating that a new layers shall be created
   *
   *  If this flag is set to true, layers not listed in the layer map a created
   *  too.
   */
  bool create_other_layers;

  /** 
   *  @brief Implementation of FormatSpecificReaderOptions
   */
  virtual FormatSpecificReaderOptions *clone () const
  {
    return new MEBESReaderOptions (*this);
  }

  /**
   *  @brief Implementation of FormatSpecificReaderOptions
   */
  virtual const std::string &format_name () const
  {
    static const std::string n = "MEBES";
    return n;
  }
};

/**
 *  @brief Generic base class of MEBES reader exceptions
 */
class MEBESReaderException
  : public ReaderException 
{
public:
  MEBESReaderException (const std::string &msg, size_t p)
    : ReaderException (tl::sprintf (tl::to_string (tr ("%s (position=%ld)")).c_str (), msg.c_str (), p))
  { }
};

/**
 *  @brief The MEBES format stream reader
 */
class MEBESReader
  : public ReaderBase
{
public: 
  /**
   *  @brief Construct a stream reader object
   *
   *  @param s The stream delegate from which to read stream data from
   */
  MEBESReader (tl::InputStream &s);

  /**  
   *  @brief Destructor
   */
  ~MEBESReader ();

  /** 
   *  @brief The basic read method 
   *
   *  This method will read the stream data and translate this to
   *  insert calls into the layout object. This will not do much
   *  on the layout object beside inserting the objects.
   *  It can be passed a layer map which tells which
   *  MEBES layer(s) to read on which logical layers.
   *  In addition, a flag can be passed that tells wheter to create 
   *  new layers. The returned map will contain all layers, the passed
   *  ones and the newly created ones.
   *
   *  @param layout The layout object to write to
   *  @param map The LayerMap object
   *  @param create true, if new layers should be created
   *  @return The LayerMap object that tells where which layer was loaded
   */
  virtual const LayerMap &read (db::Layout &layout, const db::LoadLayoutOptions &options);

  /** 
   *  @brief The basic read method (without mapping)
   *
   *  This method will read the stream data and translate this to
   *  insert calls into the layout object. This will not do much
   *  on the layout object beside inserting the objects.
   *  This version will read all input layers and return a map
   *  which tells which MEBES layer has been read into which logical
   *  layer.
   *
   *  @param layout The layout object to write to
   *  @return The LayerMap object
   */
  virtual const LayerMap &read (db::Layout &layout);

  /**
   *  @brief Accessor method to the database unit
   */
  virtual double dbu () const { return m_dbu; }

  /**
   *  @brief Format
   */
  virtual const char *format () const { return "MEBES"; }

private:
  tl::InputStream &m_stream;
  LayerMap m_layer_map;
  double m_dbu;
  MEBESReaderOptions m_options;
  tl::AbsoluteProgress m_progress;
  db::ShapeProcessor m_ep;
  std::vector<db::Polygon> m_boxes;

  void do_read (db::Layout &layout);

  void error (const std::string &txt);
  void warn (const std::string &txt, int wl = 1);

  unsigned int get_uint ();
  unsigned short get_ushort ();
  unsigned char get_ubyte ();
  void get_string (std::string &s, size_t n);
  void next_record (size_t record_len);
  void deliver_shapes (db::Shapes &output_shapes, db::Shapes &read_shapes, db::Shapes &carry, const db::Box &cell_box, const db::ICplxTrans &ct);
  void reset_delivery ();
  void deliver_background (db::Shapes &output_shapes, const db::Box &cell_box);
};

}

#endif

