
/*
  KLayout Layout Viewer
  Copyright (C) 2006-2019 Matthias Koefferlein
*/

#include "dbMEBESReader.h"
#include "dbArray.h"
#include "dbStream.h"
#include "dbSaveLayoutOptions.h"
#include "dbMetaInfo.h"

#include "tlClassRegistry.h"
#include "tlException.h"
#include "tlString.h"
#include "tlLog.h"

#include "gsiDecl.h"

#include <stdlib.h>

namespace db
{

// ---------------------------------------------------------------
//  MEBESReader declaration

class MEBESFormatDeclaration
  : public db::StreamFormatDeclaration
{
  virtual std::string format_name () const { return "MEBES"; }
  virtual std::string format_desc () const { return "MEBES"; }
  virtual std::string format_title () const { return "MEBES (e-beam pattern files)"; }
  virtual std::string file_format () const { return "MEBES files (*.MEB *.meb *.meb.gz *.MEB.gz)"; }

  virtual bool detect (tl::InputStream &stream) const 
  {
    const char *hdr = stream.get (2);
    return hdr && (hdr[1] == 0x05 || hdr[1] == 0x01);
  }

  virtual ReaderBase *create_reader (tl::InputStream &s) const 
  {
    return new db::MEBESReader (s);
  }

  virtual WriterBase *create_writer () const
  {
    return 0;
  }

  virtual bool can_read () const
  {
    return true;
  }

  virtual bool can_write () const
  {
    return false;
  }

  virtual tl::XMLElementBase *xml_reader_options_element () const
  {
    return new db::ReaderOptionsXMLElement<db::MEBESReaderOptions> ("mebes",
      tl::make_member (&db::MEBESReaderOptions::invert, "invert") +
      tl::make_member (&db::MEBESReaderOptions::subresolution, "subresolution") +
      tl::make_member (&db::MEBESReaderOptions::produce_boundary, "produce-boundary") +
      tl::make_member (&db::MEBESReaderOptions::num_stripes_per_cell, "num-stripes-per-cell") +
      tl::make_member (&db::MEBESReaderOptions::num_shapes_per_cell, "num-shapes-per-cell") +
      tl::make_member (&db::MEBESReaderOptions::data_layer, "data-layer") +
      tl::make_member (&db::MEBESReaderOptions::data_datatype, "data-datatype") +
      tl::make_member (&db::MEBESReaderOptions::data_name, "data-name") +
      tl::make_member (&db::MEBESReaderOptions::boundary_layer, "boundary-layer") +
      tl::make_member (&db::MEBESReaderOptions::boundary_datatype, "boundary-datatype") +
      tl::make_member (&db::MEBESReaderOptions::boundary_name, "boundary-name") +
      tl::make_member (&db::MEBESReaderOptions::layer_map, "layer-map") +
      tl::make_member (&db::MEBESReaderOptions::create_other_layers, "create-other-layers")
    );
  }
};

static tl::RegisteredClass<db::StreamFormatDeclaration> reader_decl (new MEBESFormatDeclaration (), 1000, "MEBES");

// ---------------------------------------------------------------
//  MEBESReader


MEBESReader::MEBESReader (tl::InputStream &s)
  : m_stream (s), 
    m_dbu (0.001), m_options (),
    m_progress (tl::to_string (tr ("Reading MEBES file")), 100000)
{
  m_progress.set_format (tl::to_string (tr ("%.0f MB")));
  m_progress.set_unit (1024 * 1024);
}

MEBESReader::~MEBESReader ()
{
  //  .. nothing yet ..
}

const LayerMap &
MEBESReader::read (db::Layout &layout, const db::LoadLayoutOptions &options)
{
#if KLAYOUT_MAJOR_VERSION > 0 || KLAYOUT_MINOR_VERSION >= 28
  init (options);
#endif

  m_options = options.get_options<MEBESReaderOptions> ();

  m_layer_map = m_options.layer_map;
  m_layer_map.prepare (layout);

  layout.start_changes ();
  do_read (layout);
  layout.end_changes ();

  return m_layer_map;
}

const LayerMap &
MEBESReader::read (db::Layout &layout)
{
  m_options = MEBESReaderOptions();

  m_layer_map = LayerMap ();
  m_layer_map.prepare (layout);

  layout.start_changes ();
  do_read (layout);
  layout.end_changes ();

  return m_layer_map;
}

inline unsigned char 
MEBESReader::get_ubyte ()
{
  const unsigned char *b = (const unsigned char *) m_stream.get (1);
  if (! b) {
    error (tl::to_string (tr ("Unexpected end-of-file (reading a single byte)")));
  }
  return *b;
}

inline unsigned short 
MEBESReader::get_ushort ()
{
  const unsigned char *b = (const unsigned char *) m_stream.get (2);
  if (! b) {
    error (tl::to_string (tr ("Unexpected end-of-file (reading a word)")));
  }
  //  TODO: this can be done more efficiently ..
  return ((unsigned short) (b[0]) << 8) | (unsigned short) (b[1]);
}

inline unsigned int 
MEBESReader::get_uint ()
{
  const unsigned char *b = (const unsigned char *) m_stream.get (4);
  if (! b) {
    error (tl::to_string (tr ("Unexpected end-of-file (reading a double word)")));
  }
  //  TODO: this can be done more efficiently ..
  return ((unsigned int) (b[0]) << 24) | ((unsigned int) (b[1]) << 16) | ((unsigned int) (b[2]) << 8) | (unsigned int) (b[3]);
}

void
MEBESReader::get_string (std::string &s, size_t n)
{
  const char *b = m_stream.get (n);
  if (! b) {
    error (tl::to_string (tr ("Unexpected end-of-file (reading a string)")));
  }
  s.assign (b, 0, n);
}

void 
MEBESReader::next_record (size_t rl)
{
  const char *b = m_stream.get (rl * ((m_stream.pos () + (rl - 1)) / rl) - m_stream.pos ());
  if (! b) {
    error (tl::to_string (tr ("Unexpected end-of-file (skipping to next block)")));
  }
}

void 
MEBESReader::error (const std::string &msg)
{
  throw MEBESReaderException (msg, m_stream.pos ());
}

void 
MEBESReader::warn (const std::string &msg, int wl)
{
#if KLAYOUT_MAJOR_VERSION > 0 || KLAYOUT_MINOR_VERSION >= 28
  if (warn_level () < wl) {
    return;
  }
#endif

  // TODO: compress
  tl::warn << msg 
           << tl::to_string (tr (" (position=")) << m_stream.pos ()
           << ")";
}

void 
MEBESReader::do_read (db::Layout &layout)
{
  unsigned int n_datafields = get_ubyte ();
  int subres = m_options.subresolution ? 16 : 1;

  unsigned char start_dwg = get_ubyte ();
  if (start_dwg != 5 && start_dwg != 1) {
    error (tl::to_string (tr ("Missing start drawing header code (5 or 1)")));
  }
  bool mode5 = (start_dwg == 1);

  if (n_datafields < 2) {
    error (tl::to_string (tr ("At least 2 data fields required")));
  }

  unsigned int dbu_int = get_uint ();
  m_dbu = 1e-8 * floor (1e8 * double (dbu_int) / double (0x10000000) + 0.5);
  layout.dbu (m_dbu);

  unsigned int stripe_height = get_ushort ();

  unsigned int cx = get_uint ();
  unsigned int cy = get_uint ();

  std::string month, day, year;

  if (mode5) {
    month = tl::sprintf ("%d", int (get_ushort ()));
    day = tl::sprintf ("%d", int (get_ushort ()));
    year = tl::sprintf ("%d", int (get_ushort ()));
  } else {
    get_string (month, 2);
    get_string (day, 2);
    get_string (year, 2);
  }

  for (unsigned int nd = 0; nd < n_datafields; ++nd) {

    if (nd == 0) {

      unsigned int words = get_ushort ();
      std::string s;
      get_string (s, words * 2);
#if defined(KLAYOUT_META_INFO_V2)
      layout.add_meta_info ("mebes_pattern_name", db::MetaInfo (tl::to_string (tr ("MEBES Pattern name")), s));
#else
      layout.add_meta_info (db::MetaInfo ("mebes_pattern_name", tl::to_string (tr ("MEBES Pattern name")), s));
#endif

    } else if (nd == 1) {

      unsigned int words;
      if (mode5) {
        words = get_uint ();
      } else {
        words = get_ushort ();
      }

      //  segment directory and others -> read over: We employ the begin_segment indicators
      const char *b = m_stream.get (words * 2);
      if (! b) {
        error (tl::to_string (tr ("Unexpected end-of-file (reading a data field)")));
      }

    } else {

      unsigned int words = get_ushort ();
      std::string s;
      get_string (s, words * 2);

      if (nd == 2) {
#if defined(KLAYOUT_META_INFO_V2)
        layout.add_meta_info ("mebes_maskshop_info", db::MetaInfo (tl::to_string (tr ("MEBES Maskshop info")), s));
#else
        layout.add_meta_info (db::MetaInfo ("mebes_maskshop_info", tl::to_string (tr ("MEBES Maskshop info")), s));
#endif
      } else {
#if defined(KLAYOUT_META_INFO_V2)
        layout.add_meta_info (tl::sprintf ("mebes_data_%d", int (nd) - 3), db::MetaInfo (tl::sprintf (tl::to_string (tr ("MEBES Data field %d")), int (nd) - 3), s));
#else
        layout.add_meta_info (db::MetaInfo (tl::sprintf ("mebes_data_%d", int (nd) - 3), tl::sprintf (tl::to_string (tr ("MEBES Data field %d")), int (nd) - 3), s));
#endif
      }

    }

  }

#if defined(KLAYOUT_META_INFO_V2)
  layout.add_meta_info ("mebes_date", db::MetaInfo (tl::to_string (tr ("MEBES file date (m/d/y)")), month + "/" + day + "/" + year));
#else
  layout.add_meta_info (db::MetaInfo ("mebes_date", tl::to_string (tr ("MEBES file date (m/d/y)")), month + "/" + day + "/" + year));
#endif

  if (mode5) {
#if defined(KLAYOUT_META_INFO_V2)
    layout.add_meta_info ("mebes_format", db::MetaInfo (tl::to_string (tr ("MEBES file format")), "MODE5"));
#else
    layout.add_meta_info (db::MetaInfo ("mebes_format", tl::to_string (tr ("MEBES file format")), "MODE5"));
#endif
  } else {
#if defined(KLAYOUT_META_INFO_V2)
    layout.add_meta_info ("mebes_format", db::MetaInfo (tl::to_string (tr ("MEBES file format")), "Reticle/Extended Address Format"));
#else
    layout.add_meta_info (db::MetaInfo ("mebes_format", tl::to_string (tr ("MEBES file format")), "Reticle/Extended Address Format"));
#endif
  }

  int frame_layer = -1;
  unsigned int layer = 0;

  db::LayerProperties dl;
  dl.layer = m_options.data_layer;
  dl.datatype = m_options.data_datatype;
  dl.name = m_options.data_name;

  db::LayerProperties frame_dl;
  frame_dl.layer = m_options.boundary_layer;
  frame_dl.datatype = m_options.boundary_datatype;
  frame_dl.name = m_options.boundary_name;

  bool shortcut = false;

#if KLAYOUT_MAJOR_VERSION > 0 || (KLAYOUT_MAJOR_VERSION == 0 && KLAYOUT_MINOR_VERSION >= 27)
  std::pair<bool, unsigned int> ll = m_layer_map.first_logical (dl);
#else
  std::pair<bool, unsigned int> ll = m_layer_map.logical (dl);
#endif
  if (! ll.first) {

    //  no need to read anything - we cannot create layers
    if (m_options.create_other_layers) {

      //  and create the layer
      layer = layout.insert_layer (dl);
      m_layer_map.map (dl, layer);

    } else {
      shortcut = true;
    }

  } else {
    layer = ll.second;
  }

  if (m_options.produce_boundary) {

#if KLAYOUT_MAJOR_VERSION > 0 || (KLAYOUT_MAJOR_VERSION == 0 && KLAYOUT_MINOR_VERSION >= 27)
    ll = m_layer_map.first_logical (frame_dl);
#else
    ll = m_layer_map.logical (frame_dl);
#endif
    if (! ll.first) {

      //  no need to read anything - we cannot create layers
      if (m_options.create_other_layers) {

        //  and create the layer
        frame_layer = layout.insert_layer (frame_dl);
        m_layer_map.map (frame_dl, frame_layer);

      }

    } else {
      frame_layer = ll.second;
    }

  }

  size_t record_len = 2048;

  //  Note: is this different in other modes?
  size_t stripe_width = 32768;

  unsigned short cmd;
  unsigned short wcmd;
  unsigned int segment_number = 1;
  unsigned int stripe_number = 1;
  db::Vector stripe_offset;

  const char *top_cell_name = "MEBES_TOP";
  db::Cell &top_cell = layout.cell (layout.is_valid_cell_index (m_options.top_cell_index) ? m_options.top_cell_index : layout.add_cell (top_cell_name));

  //  create the frame layer if required
  if (frame_layer >= 0) {
    db::Box box (db::Point (0, 0), db::Point (cx, cy));
    top_cell.shapes ((unsigned int) frame_layer).insert (box);
  }

  //  if we don't need to read the actual shapes, stop now
  if (shortcut) {
    return;
  }

  //  there is no end-of-record marker for MODE5 header files.
  if (mode5) {
    next_record (record_len);
  }

  db::Cell *cell = 0;
  unsigned int cell_seg = 0;
  unsigned int cell_start_stripe = 0;
  db::Vector cell_offset;
  db::ICplxTrans cell_trans;
  db::Shapes shapes (layout.is_editable ());
  db::Shapes carry;

  reset_delivery ();

  //  actually read the file
  while (true) {

    wcmd = get_ushort ();
    cmd = wcmd & 0xff;

    if (cmd == 4) {

      break;

    } else if (cmd == 9) {

      next_record (record_len);

    } else if (cmd == 10 && !mode5) {

      segment_number = (wcmd >> 8);
      if (segment_number == 0) {
        error (tl::to_string (tr ("Segment number of 0 is not allowed")));
      }

    } else if (cmd == 12 && mode5) {

      segment_number = get_uint ();
      if (segment_number == 0) {
        error (tl::to_string (tr ("Segment number of 0 is not allowed")));
      }

    } else if ((cmd == 2 && !mode5) || (cmd == 13 && mode5)) {

      stripe_number = (cmd == 13) ? get_uint () : get_ushort ();
      if (stripe_number == 0) {
        error (tl::to_string (tr ("Stripe number of 0 is not allowed")));
      }

      db::Coord sx = db::Coord ((int (segment_number) - 1) * stripe_width);
      db::Coord sy = db::Coord ((int (stripe_number) - 1) * stripe_height);
      stripe_offset = db::Vector (sx, sy);

      if (! cell || segment_number != cell_seg || 
          ((m_options.num_stripes_per_cell != 0 && ((stripe_number - 1) / m_options.num_stripes_per_cell) != (cell_start_stripe - 1) / m_options.num_stripes_per_cell) &&
           (m_options.num_shapes_per_cell == 0 || shapes.size () > size_t (m_options.num_shapes_per_cell)))) {

        db::ICplxTrans tp = cell_trans;

        if (cell) {
          db::Box cb; 
          if (segment_number != cell_seg) {
            cb = db::Box (0, 0, db::Coord (stripe_width * subres), db::Coord (cy * subres));
          } else {
            cb = db::Box (0, 0, db::Coord (stripe_width * subres), db::Coord ((stripe_number - cell_start_stripe) * stripe_height * subres));
          }
          cb &= db::Box (db::Point (-cell_offset.x () * subres, -cell_offset.y () * subres), db::Point ((cx - cell_offset.x ()) * subres, (cy - cell_offset.y ()) * subres));
          deliver_shapes (cell->shapes (layer), shapes, carry, cb, cell_trans);
        }

        cell = & layout.cell (layout.add_cell (tl::sprintf ("%d_%d", segment_number, stripe_number).c_str ()));
        cell_seg = segment_number;
        cell_start_stripe = stripe_number;
        cell_offset = stripe_offset;

        //  since MEBES shapes are basically given at a resolution of 1/16 database unit we
        //  store the shapes 16 times larger and compensate this by placing the stripe cells with a
        //  demagnification of 1/16th.
        db::CellInstArray ci (db::CellInst (cell->cell_index ()), db::Trans (0, false, stripe_offset), 1.0, 1.0 / double (subres));
        cell_trans = ci.complex_trans ();
        top_cell.insert (ci);

        //  some shapes may be remaining from the previous tile
        //  These need to be transformed into the current cell
        if (! carry.empty ()) {
          db::ICplxTrans t = cell_trans.inverted () * tp;
          shapes.insert_transformed (carry, t);
        }

      }

      stripe_offset -= cell_offset;
      
      unsigned int nx = 0, ny = 0;
      unsigned int ax = 0, ay = 0;
      unsigned int dx = 0, dy = 0;
      bool repeated = false;

      while (true) {

        m_progress.set (m_stream.pos ());

        wcmd = get_ushort ();
        cmd = wcmd & 0x3f;

        if (cmd == 8) {
          break;
        } else if (cmd == 9) {
          next_record (record_len);
        } else if (cmd == 16) {

          //  Rectangle
          db::Coord h = subres * ((wcmd >> 6) + 1);
          db::Coord w = subres * get_ushort ();
          db::Coord x = subres * get_ushort ();
          db::Coord y = subres * get_ushort ();

          //  Slight optimization: combine sliced boxes if they
          //  join seamlessly
          if (repeated) {

            if (ny > 1 && h == db::Coord (dy * subres)) {
              h *= ny;
              ny = 1;
            }

            if (nx > 1 && w == db::Coord (dx * subres)) {
              w *= nx;
              nx = 1;
            }

            repeated = (nx > 1 || ny > 1);

          }

          if (repeated) {

            if (!mode5) {
              x = y = 0;
            }

            x += subres * (ax + stripe_offset.x ());
            y += subres * (ay + stripe_offset.y ());

            //  A polygon pointer is most efficiently created from a normalized shape
            db::Box box (db::Point (0, 0), db::Point (w, h));

            if (x >= std::numeric_limits <short>::min () && x <= std::numeric_limits <short>::max () && 
                y >= std::numeric_limits <short>::min () && y <= std::numeric_limits <short>::max () && 
                x + w >= std::numeric_limits <short>::min () && x + w <= std::numeric_limits <short>::max () && 
                y + h >= std::numeric_limits <short>::min () && y + h <= std::numeric_limits <short>::max ()) {
              db::ShortBox box (x, y, x + w, y + h);
              shapes.insert (db::array<db::ShortBox, db::UnitTrans> (box, db::UnitTrans (), layout.array_repository (), db::Vector (dx * subres, 0), db::Vector (0, dy * subres), nx, ny));
            } else {
              db::Box box (db::Point (x, y), db::Point (x + w, y + h));
              shapes.insert (db::array<db::Box, db::UnitTrans> (box, db::UnitTrans (), layout.array_repository (), db::Vector (dx * subres, 0), db::Vector (0, dy * subres), nx, ny));
            }

            repeated = false;

          } else {

            x += subres * stripe_offset.x ();
            y += subres * stripe_offset.y ();

            if (x >= std::numeric_limits <short>::min () && x <= std::numeric_limits <short>::max () && 
                y >= std::numeric_limits <short>::min () && y <= std::numeric_limits <short>::max () && 
                x + w >= std::numeric_limits <short>::min () && x + w <= std::numeric_limits <short>::max () && 
                y + h >= std::numeric_limits <short>::min () && y + h <= std::numeric_limits <short>::max ()) {
              db::ShortBox box (x, y, x + w, y + h);
              shapes.insert (box);
            } else {
              db::Box box (db::Point (x, y), db::Point (x + w, y + h));
              shapes.insert (box);
            }

          }

        } else if (cmd >= 17 && cmd <= 20) {

          db::Coord h, w, x1, y1, dx1, dx2;

          if (cmd == 20) {

            //  Trapezoid 3
            h = subres * db::Coord ((wcmd >> 6) + 1);
            w = get_ushort ();
            x1 = get_ushort ();
            unsigned short s1 = get_ushort ();
            y1 = subres * db::Coord (s1 & 0x03ff);
            dx1 = get_ushort ();
            dx2 = get_ushort ();
            unsigned short s2 = get_ushort ();
            dx1 += db::Coord (s2 & 0xfc00) << 6;
            if (s2 & 0x8000) {
              dx1 -= 0x400000;
            }
            dx2 += db::Coord (s1 & 0xfc00) << 6;
            if (s1 & 0x8000) {
              dx2 -= 0x400000;
            }
            x1 += db::Coord (s2 & 0x03e0) << 11;
            w += (s2 & 0x001f) << 16;

          } else {

            //  Parallelogram, Trapezoid 1 or Trapezoid 2
            h = subres * ((wcmd >> 6) + 1);
            w = get_ushort ();
            x1 = get_ushort ();
            y1 = subres * get_ushort ();
            dx1 = get_ushort ();
            unsigned short s = get_ushort ();
            dx1 += db::Coord (s & 0xfc00) << 6;
            if (s & 0x8000) {
              dx1 -= 0x400000;
            }
            x1 += db::Coord (s & 0x03e0) << 11;
            w += db::Coord (s & 0x001f) << 16;

            dx2 = dx1;
            if (cmd == 18) {
              dx2 = 0;
            } else if (cmd == 19) {
              dx1 = 0;
            }

          }

          if (subres == 1) {
            if (w % 16 != 0) {
              error (tl::sprintf (tl::to_string (tr ("Sub-resolution shape (width=%d)")).c_str (), w));
            }
            w /= 16;
            if (x1 % 16 != 0) {
              error (tl::sprintf (tl::to_string (tr ("Sub-resolution shape (x1=%d)")).c_str (), x1));
            }
            x1 /= 16;
            if (dx1 % 16 != 0) {
              error (tl::sprintf (tl::to_string (tr ("Sub-resolution shape (dx1=%d)")).c_str (), dx1));
            }
            dx1 /= 16;
            if (dx2 % 16 != 0) {
              error (tl::sprintf (tl::to_string (tr ("Sub-resolution shape (dx2=%d)")).c_str (), dx2));
            }
            dx2 /= 16;
          }

          if (repeated) {

            if (! mode5) {
              //  in reticle mode, the integer part is superseeded by repetition specification
              x1 = x1 % subres; 
              y1 = y1 % subres;
            }

            x1 += subres * stripe_offset.x ();
            y1 += subres * stripe_offset.y ();

            //  A polygon pointer is most efficiently created from a normalized shape
            db::Point pts[4] = {
              db::Point (0, 0),
              db::Point (dx1, 0 + h),
              db::Point (dx2 + w, 0 + h),
              db::Point (w, 0)
            };

            db::SimplePolygon poly;
            poly.assign_hull (pts, pts + 4);

            //  there is no box array currently, so we use a polygon array
            db::SimplePolygonPtr poly_ptr (poly, layout.shape_repository ());
            shapes.insert (db::array<db::SimplePolygonPtr, db::Disp> (poly_ptr, db::Disp (db::Vector (ax * subres + x1, ay * subres + y1)), layout.array_repository (), db::Vector (dx * subres, 0), db::Vector (0, dy * subres), nx, ny));
            repeated = false;

          } else {

            x1 += subres * stripe_offset.x ();
            y1 += subres * stripe_offset.y ();

            db::Point pts[4] = {
              db::Point (x1, y1),
              db::Point (x1 + dx1, y1 + h),
              db::Point (x1 + dx2 + w, y1 + h),
              db::Point (x1 + w, y1)
            };

            db::SimplePolygon poly;
            poly.assign_hull (pts, pts + 4);

            shapes.insert (db::SimplePolygonRef (poly, layout.shape_repository ()));

          }

        } else if ((cmd == 14 && !mode5) || (cmd == 21 && mode5)) {

          //  Data compaction (repeat-setup)
          if (mode5) {
            ax = ay = 0;
          } else {
            ax = get_ushort ();
            ay = get_ushort ();
          }
          dx = get_ushort ();
          dy = get_ushort ();
          nx = get_ushort ();
          ny = get_ushort ();
          repeated = (nx > 1 || ny > 1);

        } else if (cmd == 15 && !mode5) {
          //  Data compaction (repeat-execute)
        } else {
          error (tl::to_string (tr ("Unexpected command (expected figure data)")));
        }

      }

    } else {
      error (tl::to_string (tr ("Unexpected command (expected start of segment or end of drawing)")));
    }

  }

  if (cell) {
    db::Box cb = db::Box (0, 0, db::Coord (stripe_width * subres), db::Coord (cy * subres)) & db::Box (db::Point (-cell_offset.x () * subres, -cell_offset.y () * subres), db::Point ((cx - cell_offset.x ()) * subres, (cy - cell_offset.y ()) * subres));
    deliver_shapes (cell->shapes (layer), shapes, carry, cb, cell_trans);
    cell = 0;
  }

  deliver_background (top_cell.shapes (layer), db::Box (0, 0, cx, cy));
}

void 
MEBESReader::reset_delivery ()
{
  m_boxes.clear ();
}

void
MEBESReader::deliver_background (db::Shapes &output_shapes, const db::Box &cell_box) 
{
  if (m_options.invert) {

    db::EdgeProcessor ep;

    std::vector<db::Polygon> whole_box;
    whole_box.push_back (db::Polygon (cell_box));

    std::vector<db::Polygon> background;

    ep.boolean (whole_box, m_boxes, background, db::BooleanOp::ANotB);

    for (std::vector<db::Polygon>::const_iterator b = background.begin (); b != background.end (); ++b) {
      output_shapes.insert (*b);
    }

  }
}

void
MEBESReader::deliver_shapes (db::Shapes &output_shapes, db::Shapes &read_shapes, db::Shapes &carry, const db::Box &cell_box, const db::ICplxTrans &ct)
{
  if (m_options.invert) {

    m_boxes.push_back (db::Polygon (ct * cell_box));

    db::Layout l;
    l.dbu (m_dbu);
    db::Cell &c = l.cell (l.add_cell ());
    l.insert_layer (0);
    l.insert_layer (1);
    c.shapes (0).insert (cell_box);
    //  TODO: swap does not work!
    c.shapes (1) = read_shapes;
    read_shapes.clear ();

    output_shapes.clear ();
    m_ep.boolean (l, c, 0, l, c, 1, output_shapes, db::BooleanOp::ANotB, false, false /*with_holes*/);

    //  shapes reaching out of the tile are kept and processed with the next tile
    carry.clear ();
    m_ep.boolean (l, c, 0, l, c, 1, carry, db::BooleanOp::BNotA, false, false /*with_holes*/);

  } else {
    //  TODO: swap does not work!
    output_shapes = read_shapes;
    read_shapes.clear ();
  }
}

//  provide a symbol to force linking against
int force_link_MEBES = 0;

}

