
/*
  KLayout Layout Viewer
  Copyright (C) 2006-2019 Matthias Koefferlein
*/

#ifndef HDR_dbMEBESCompression
#define HDR_dbMEBESCompression

#include "dbPluginCommon.h"

#include "dbVector.h"

namespace db
{

/**
 *  @brief A struct representing an MEBES array
 */
struct TrapezoidArraySpec
{
  /**
   *  @brief Default constructor
   *  This constructor creates a single-instance array at 0,0
   */
  TrapezoidArraySpec ();

  /**
   *  @brief Generic constructor
   *  @param _offset The first instance
   *  @param _nx The number of columns
   *  @param _ny The number of rows
   *  @param _dx The column distance
   *  @param _dy The row distance
   */
  TrapezoidArraySpec (const db::Vector &_offset, size_t _nx, size_t _ny, db::Coord _dx, db::Coord _dy);

  db::Vector offset;
  size_t nx, ny;
  db::Coord dx, dy;
};

/**
 *  @brief The compressor class
 *
 *  The scenario for using this class it to feed it displacements
 *  by using "add", then call "compress" and finally fetching the remaining
 *  single displacements and arrays.
 */
class DB_PLUGIN_PUBLIC DisplacementCompressor
{
public:
  typedef std::vector<db::Vector> disp_vector;
  typedef disp_vector::const_iterator disp_iterator;
  typedef std::vector<TrapezoidArraySpec> array_vector;
  typedef array_vector::const_iterator array_iterator;

  /**
   *  @brief Constructor
   */
  DisplacementCompressor ();

  /**
   *  @brief Clears the data contained in this compressor
   */
  void clear ();

  /**
   *  @brief Adds displacements
   */
  template <class I>
  void add (I begin, I end)
  {
    m_displacements.reserve (m_displacements.size () + end - begin);
    m_displacements.insert (m_displacements.end (), begin, end);
  }

  /**
   *  @brief Adds a single displacement
   */
  void add (const db::Vector &d)
  {
    m_displacements.push_back (d);
  }

  /**
   *  @briefs Swaps the given displacement vector into ourself
   */
  void swap_displacements (disp_vector &displacements)
  {
    m_displacements.swap (displacements);
  }

  /**
   *  @briefs Swaps the given arrays vector into ourself
   */
  void swap_arrays (array_vector &arrays)
  {
    m_arrays.swap (arrays);
  }

  /**
   *  @brief Performs the compression
   *  @param compression_level A value indicating the strength of compression (0: none .. 10: max)
   *  @param subresolution_fracturing_enabled If true, certain constraints apply (i.e. x pitch must be a multiple of 16)
   */
  void compress (unsigned int compression_level, bool subresolution_fracturing_enabled);

  /**
   *  @brief Fetches the arrays (begin iterator)
   */
  array_iterator begin_arrays () const
  {
    return m_arrays.begin ();
  }

  /**
   *  @brief Fetches the arrays (end iterator)
   */
  array_iterator end_arrays () const
  {
    return m_arrays.end ();
  }

  /**
   *  @brief Fetches the remaining single displacements (begin iterator)
   */
  disp_iterator begin_displacements () const
  {
    return m_displacements.begin ();
  }

  /**
   *  @brief Fetches the remaining single displacements (end iterator)
   */
  disp_iterator end_displacements () const
  {
    return m_displacements.end ();
  }

private:
  disp_vector m_displacements;
  array_vector m_arrays;
};

}

#endif
